<?php

namespace Drupal\nanobanana_editor\Form;

use Drupal\ai\AiProviderPluginManager;
use Drupal\ai\OperationType\GenericType\ImageFile;
use Drupal\ai\OperationType\ImageToImage\ImageToImageInput;
use Drupal\ai_provider_nanobanana\OperationType\MultiImageToImageInput;
use Drupal\Component\Datetime\TimeInterface;
use Drupal\Component\Uuid\UuidInterface;
use Drupal\Core\Ajax\AjaxResponse;
use Drupal\Core\Ajax\HtmlCommand;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\File\FileExists;
use Drupal\Core\File\FileSystemInterface;
use Drupal\Core\File\FileUrlGeneratorInterface;
use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormBuilderInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\TempStore\PrivateTempStoreFactory;
use Drupal\media\MediaInterface;
use Psr\Log\LoggerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Form for editing media images with NanoBanana AI.
 */
final class NanoBananaEditorForm extends FormBase {

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The AI provider plugin manager.
   *
   * @var \Drupal\ai\AiProviderPluginManager
   */
  protected $aiProviderManager;

  /**
   * The file system service.
   *
   * @var \Drupal\Core\File\FileSystemInterface
   */
  protected $fileSystem;

  /**
   * The config factory service.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * The logger service.
   *
   * @var \Psr\Log\LoggerInterface
   */
  protected $logger;

  /**
   * The form builder service.
   *
   * @var \Drupal\Core\Form\FormBuilderInterface
   */
  protected $formBuilder;

  /**
   * The file URL generator service.
   *
   * @var \Drupal\Core\File\FileUrlGeneratorInterface
   */
  protected $fileUrlGenerator;

  /**
   * The datetime.time service.
   *
   * @var \Drupal\Component\Datetime\TimeInterface
   */
  protected $dateTime;

  /**
   * The private tempstore.
   *
   * @var \Drupal\Core\TempStore\PrivateTempStore
   */
  protected $tempStore;

  /**
   * The UUID generator.
   *
   * @var \Drupal\Component\Uuid\UuidInterface
   */
  protected $uuid;

  /**
   * Constructs a new NanoBananaEditorForm.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   * @param \Drupal\ai\AiProviderPluginManager $ai_provider_manager
   *   The AI provider plugin manager.
   * @param \Drupal\Core\File\FileSystemInterface $file_system
   *   The file system service.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory service.
   * @param \Psr\Log\LoggerInterface $logger
   *   The logger service.
   * @param \Drupal\Core\Form\FormBuilderInterface $form_builder
   *   The form builder service.
   * @param \Drupal\Core\File\FileUrlGeneratorInterface $file_url_generator
   *   The file URL generator service.
   * @param \Drupal\Component\Datetime\TimeInterface $date_time
   *   The datetime.time service.
   * @param \Drupal\Core\TempStore\PrivateTempStoreFactory $temp_store_factory
   *   The private tempstore factory.
   * @param \Drupal\Component\Uuid\UuidInterface $uuid
   *   The UUID generator.
   */
  public function __construct(EntityTypeManagerInterface $entity_type_manager, AiProviderPluginManager $ai_provider_manager, FileSystemInterface $file_system, ConfigFactoryInterface $config_factory, LoggerInterface $logger, FormBuilderInterface $form_builder, FileUrlGeneratorInterface $file_url_generator, TimeInterface $date_time, PrivateTempStoreFactory $temp_store_factory, UuidInterface $uuid) {
    $this->entityTypeManager = $entity_type_manager;
    $this->aiProviderManager = $ai_provider_manager;
    $this->fileSystem = $file_system;
    $this->configFactory = $config_factory;
    $this->logger = $logger;
    $this->formBuilder = $form_builder;
    $this->fileUrlGenerator = $file_url_generator;
    $this->dateTime = $date_time;
    $this->tempStore = $temp_store_factory->get('nanobanana_editor');
    $this->uuid = $uuid;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new self(
      $container->get('entity_type.manager'),
      $container->get('ai.provider'),
      $container->get('file_system'),
      $container->get('config.factory'),
      $container->get('logger.factory')->get('nanobanana_editor'),
      $container->get('form_builder'),
      $container->get('file_url_generator'),
      $container->get('datetime.time'),
      $container->get('tempstore.private'),
      $container->get('uuid')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'nanobanana_editor_form';
  }

  /**
   * Value callback to prevent file processing.
   *
   * @return null
   *   Always returns NULL to prevent file processing.
   */
  public static function valueCallback() {
    return NULL;
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state, ?MediaInterface $media = NULL) {
    if (!$media) {
      return $form;
    }

    // Disable form caching to prevent serialization issues with file uploads.
    $form_state->setCached(FALSE);
    $form_state->disableCache();

    // Reload generated image from tempstore when returning via non-AJAX submit.
    $token = $form_state->get('generated_image_token');
    if (!$token) {
      $user_input = $form_state->getUserInput();
      if (!empty($user_input['generated_image_token'])) {
        $token = $user_input['generated_image_token'];
      }
    }
    if ($token) {
      $form_state->set('generated_image_token', $token);
    }
    if (!$form_state->get('generated_image') && $token) {
      $stored_image = $this->tempStore->get($token);
      if ($stored_image) {
        $form_state->set('generated_image', $stored_image);
      }
    }

    // Store the media entity in form state.
    $form_state->set('media', $media);

    // Get the source field and image file.
    $source_field = $media->getSource()->getConfiguration()['source_field'];
    if (!$media->hasField($source_field) || $media->get($source_field)->isEmpty()) {
      $form['error'] = [
        '#markup' => $this->t('No image found for this media.'),
      ];
      return $form;
    }

    /** @var \Drupal\file\FileInterface $file */
    $file = $media->get($source_field)->entity;
    if (!$file) {
      $form['error'] = [
        '#markup' => $this->t('Image file not found.'),
      ];
      return $form;
    }

    $form['#prefix'] = '<div id="nanobanana-editor-wrapper" class="nanobanana-editor-layout">';
    $form['#suffix'] = '</div>';

    // Main content container (left side).
    $form['main_content'] = [
      '#type' => 'container',
      '#attributes' => ['class' => ['nanobanana-main-content']],
    ];

    // Image preview container.
    $form['main_content']['preview_container'] = [
      '#type' => 'container',
      '#attributes' => ['id' => 'image-preview-container', 'class' => ['nanobanana-preview']],
    ];

    // Check if we have a generated image in form state.
    $generated_image = $form_state->get('generated_image');
    if ($generated_image) {
      // Display the generated image as data URL.
      $data_uri = 'data:image/png;base64,' . base64_encode($generated_image);
      $form['main_content']['preview_container']['preview'] = [
        '#type' => 'html_tag',
        '#tag' => 'img',
        '#attributes' => [
          'src' => $data_uri,
          'alt' => $this->t('Generated image'),
          'class' => ['nanobanana-preview-image'],
        ],
      ];
    }
    else {
      // Display the original image.
      $image_url = $this->fileUrlGenerator->generateAbsoluteString($file->getFileUri());
      $form['main_content']['preview_container']['preview'] = [
        '#type' => 'html_tag',
        '#tag' => 'img',
        '#attributes' => [
          'src' => $image_url,
          'alt' => $this->t('Current image'),
          'class' => ['nanobanana-preview-image'],
        ],
      ];
    }

    // Prompt textarea in main content.
    $form['main_content']['prompt'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Prompt'),
      '#description' => $this->t('Describe the changes you want to make to the image. Be specific about what you want (e.g., "Make the background blue", "Add a sunset effect", "Convert to black and white").'),
      '#placeholder' => $this->t('E.g., Make the image look like a watercolor painting'),
      '#required' => TRUE,
      '#rows' => 3,
    ];

    // Status messages container in main content.
    $form['main_content']['messages'] = [
      '#type' => 'container',
      '#attributes' => ['id' => 'nanobanana-messages', 'class' => ['nanobanana-messages']],
    ];

    if ($form_state->get('generated_image_token')) {
      $form['generated_image_token'] = [
        '#type' => 'hidden',
        '#value' => $form_state->get('generated_image_token'),
      ];
    }

    // Sidebar container (right side).
    $form['sidebar'] = [
      '#type' => 'fieldset',
      '#attributes' => ['class' => ['nanobanana-sidebar']],
    ];

    // Model selection field in sidebar.
    $form['sidebar']['model'] = [
      '#type' => 'select',
      '#title' => $this->t('Model'),
      '#description' => $this->t('Select the AI model to use for image generation.'),
      '#options' => [
        'gemini-2.5-flash-image' => $this->t('Gemini 2.5 Flash (Fast, supports up to 2 additional images)'),
        'gemini-3-pro-image-preview' => $this->t('Gemini 3 Pro (High quality, supports up to 13 additional images)'),
      ],
      '#default_value' => 'gemini-2.5-flash-image',
      '#required' => TRUE,
      '#ajax' => [
        'callback' => '::updateModelDependentFieldsCallback',
        'wrapper' => 'model-dependent-fields-wrapper',
        'event' => 'change',
      ],
    ];

    // Get the selected model (from form state or default).
    $selected_model = $form_state->getValue('model') ?: 'gemini-2.5-flash-image';
    $max_files = $selected_model === 'gemini-3-pro-image-preview' ? 13 : 2;
    $model_name = $selected_model === 'gemini-3-pro-image-preview' ? 'Gemini 3 Pro' : 'Gemini 2.5 Flash';
    $is_gemini_3_pro = $selected_model === 'gemini-3-pro-image-preview';

    // Model-dependent fields wrapper (with AJAX wrapper) in sidebar.
    $form['sidebar']['model_dependent_fields_wrapper'] = [
      '#type' => 'container',
      '#attributes' => ['id' => 'model-dependent-fields-wrapper'],
    ];

    // Additional images upload field for composite generation.
    $form['sidebar']['model_dependent_fields_wrapper']['additional_images'] = [
      '#type' => 'file',
      '#title' => $this->t('Additional Images (Optional)'),
      '#description' => $this->t('Upload additional images for multi-image composition. You can select multiple files at once (hold Ctrl/Cmd while clicking). @model supports up to @max additional images.', [
        '@model' => $model_name,
        '@max' => $max_files,
      ]),
      '#multiple' => TRUE,
      '#name' => 'files[additional_images][]',
      '#attributes' => [
        'accept' => 'image/png,image/jpeg,image/jpg',
        'multiple' => 'multiple',
      ],
      '#attached' => [
        'library' => ['nanobanana_editor/image_preview'],
      ],
      '#required' => FALSE,
      '#process' => [],
      '#value_callback' => [get_class($this), 'valueCallback'],
    ];

    // Image Size field (only for Gemini 3 Pro).
    if ($is_gemini_3_pro) {
      $form['sidebar']['model_dependent_fields_wrapper']['image_size'] = [
        '#type' => 'select',
        '#title' => $this->t('Image Size'),
        '#description' => $this->t('Output image size (Gemini 3 Pro only). Higher resolution takes longer to generate.'),
        '#options' => [
          '' => $this->t('Default'),
          '1K' => $this->t('1K'),
          '2K' => $this->t('2K'),
          '4K' => $this->t('4K'),
        ],
        '#default_value' => '',
        '#required' => FALSE,
      ];
    }

    // Aspect ratio select field in sidebar.
    $form['sidebar']['aspect_ratio'] = [
      '#type' => 'select',
      '#title' => $this->t('Aspect Ratio'),
      '#description' => $this->t('Select the aspect ratio for the generated image. Default will maintain the original aspect ratio.'),
      '#options' => [
        '' => $this->t('Default'),
        '1:1' => $this->t('1:1 (Square)'),
        '3:4' => $this->t('3:4 (Portrait)'),
        '4:3' => $this->t('4:3 (Landscape)'),
        '9:16' => $this->t('9:16 (Vertical)'),
        '16:9' => $this->t('16:9 (Widescreen)'),
      ],
      '#default_value' => '',
      '#required' => FALSE,
    ];

    // Styles select field in sidebar.
    $config = $this->configFactory->get('nanobanana_editor.settings');
    $styles = $config->get('styles') ?: [];
    $style_options = ['' => $this->t('- None -')];
    foreach ($styles as $index => $style) {
      $style_options[$index] = $style['name'];
    }

    $form['sidebar']['style'] = [
      '#type' => 'select',
      '#title' => $this->t('Style'),
      '#description' => $this->t('Select a style to apply to the edited image. The style prompt will be added to your prompt.'),
      '#options' => $style_options,
      '#default_value' => '',
      '#required' => FALSE,
    ];

    // Footer.
    $form['footer'] = [
      '#type' => 'container',
      '#attributes' => ['class' => ['nanobanana-footer']],
    ];

    // Actions in footer.
    $form['footer']['actions'] = [
      '#type' => 'actions',
    ];

    $form['footer']['actions']['generate'] = [
      '#type' => 'submit',
      '#value' => $this->t('Generate'),
      '#attributes' => ['class' => ['button']],
      '#ajax' => [
        'callback' => '::generateCallback',
        'wrapper' => 'nanobanana-editor-wrapper',
        'progress' => [
          'type' => 'throbber',
          'message' => $this->t('Generating image...'),
        ],
      ],
    ];

    // Only show save button if we have a generated image.
    if ($generated_image) {
      $form['footer']['actions']['save'] = [
        '#type' => 'submit',
        '#value' => $this->t('Save'),
        '#attributes' => ['class' => ['button']],
        '#submit' => ['::saveCallback'],
      ];
    }

    $form['footer']['actions']['cancel'] = [
      '#type' => 'submit',
      '#value' => $this->t('Cancel'),
      '#attributes' => ['class' => ['button']],
      '#submit' => ['::cancelCallback'],
      '#limit_validation_errors' => [],
    ];

    // Attach the library.
    $form['#attached']['library'][] = 'nanobanana_editor/editor';

    // Add validation handler.
    $form['#validate'][] = '::validateAdditionalImages';

    return $form;
  }

  /**
   * AJAX callback to update model-dependent fields.
   */
  public function updateModelDependentFieldsCallback(array &$form, FormStateInterface $form_state) {
    return $form['sidebar']['model_dependent_fields_wrapper'];
  }

  /**
   * Validation handler to check the number of additional images.
   */
  public function validateAdditionalImages(array &$form, FormStateInterface $form_state) {
    // Only validate when the generate button is clicked, not on model change.
    $triggering_element = $form_state->getTriggeringElement();
    if (isset($triggering_element['#name']) && $triggering_element['#name'] !== 'op') {
      // This is an AJAX callback for model change, skip validation.
      return;
    }

    // Count valid uploaded files using $_FILES to avoid serialization issues.
    $file_count = 0;

    if (isset($_FILES['files']['name']['additional_images'])) {
      $file_data = $_FILES['files']['name']['additional_images'];

      if (is_array($file_data)) {
        foreach ($file_data as $index => $file_name_data) {
          $error_data = $_FILES['files']['error']['additional_images'][$index] ?? NULL;

          // Handle extra level of nesting.
          if (is_array($error_data) && isset($error_data[0])) {
            $error = $error_data[0];
          }
          else {
            $error = $error_data;
          }

          if ($error === UPLOAD_ERR_OK) {
            $file_count++;
          }
        }
      }
    }

    if ($file_count > 0) {
      // Get the selected model.
      $model = $form_state->getValue('model');
      if (empty($model)) {
        return;
      }

      $max_files = $model === 'gemini-3-pro-image-preview' ? 13 : 2;

      if ($file_count > $max_files) {
        $model_name = $model === 'gemini-3-pro-image-preview' ? 'Gemini 3 Pro' : 'Gemini 2.5 Flash';
        $form_state->setErrorByName('model_dependent_fields_wrapper][additional_images', $this->t('You uploaded @count additional images, but @model only supports up to @max additional images. Please select fewer files.', [
          '@count' => $file_count,
          '@model' => $model_name,
          '@max' => $max_files,
        ]));
      }
    }

  }

  /**
   * AJAX callback for the Generate button.
   */
  public function generateCallback(array &$form, FormStateInterface $form_state) {

    $response = new AjaxResponse();

    // Check if there are validation errors.
    if ($form_state->hasAnyErrors()) {
      // Return the form with errors displayed.
      $errors = $form_state->getErrors();
      $error_message = implode('<br>', $errors);
      $response->addCommand(new HtmlCommand('#nanobanana-messages', '<div class="messages messages--error">' . $error_message . '</div>'));
      return $response;
    }

    /** @var \Drupal\media\MediaInterface $media */
    $media = $form_state->get('media');
    $prompt = $form_state->getValue('prompt');
    $style_index = $form_state->getValue('style');

    // Get configuration.
    $config = $this->configFactory->get('nanobanana_editor.settings');

    // Start with system instructions if available.
    $final_prompt = '';
    $system_instructions = $config->get('system_instructions');
    if (!empty($system_instructions)) {
      $final_prompt = $system_instructions . ' ';
    }

    // Add user prompt.
    $final_prompt .= $prompt;

    // Merge style prompt if a style is selected.
    if ($style_index !== '') {
      $styles = $config->get('styles') ?: [];
      if (isset($styles[$style_index])) {
        $style_prompt = $styles[$style_index]['prompt'];
        $final_prompt .= ' ' . $style_prompt;
      }
    }

    $this->logger->info('Final prompt: @prompt', ['@prompt' => $final_prompt]);

    // Use the final merged prompt.
    $prompt = $final_prompt;

    // Get the source field and image file.
    $source_field = $media->getSource()->getConfiguration()['source_field'];
    /** @var \Drupal\file\FileInterface $file */
    $file = $media->get($source_field)->entity;

    try {
      // Load the AI provider.
      $provider = $this->aiProviderManager->createInstance('nanobanana');

      // Get the selected aspect ratio (defaults to empty string).
      $aspect_ratio = $form_state->getValue('aspect_ratio') ?: '';

      // Set the configuration with aspectRatio and imageSize.
      $configuration = [
        'aspectRatio' => $aspect_ratio,
      ];

      // Add imageSize if provided (Gemini 3 Pro only).
      $image_size = $form_state->getValue('image_size');
      if (!empty($image_size)) {
        $configuration['imageSize'] = $image_size;
      }

      $provider->setConfiguration($configuration);

      // Read the image file.
      $image_path = $this->fileSystem->realpath($file->getFileUri());
      $image_binary = file_get_contents($image_path);

      // Create ImageFile object for the main image.
      $image_file = new ImageFile($image_binary, $file->getMimeType(), $file->getFilename());

      // Collect all uploaded images.
      $image_files = [$image_file];

      // Check for additional images using $_FILES to avoid serialization.
      if (isset($_FILES['files']['name']['additional_images'])) {
        $file_data = $_FILES['files']['name']['additional_images'];

        if (is_array($file_data)) {
          foreach ($file_data as $index => $file_name_data) {
            $tmp_name_data = $_FILES['files']['tmp_name']['additional_images'][$index] ?? NULL;
            $file_type_data = $_FILES['files']['type']['additional_images'][$index] ?? NULL;
            $error_data = $_FILES['files']['error']['additional_images'][$index] ?? NULL;

            // Handle extra level of nesting.
            if (is_array($file_name_data) && isset($file_name_data[0])) {
              $file_name = $file_name_data[0];
              $tmp_name = $tmp_name_data[0] ?? NULL;
              $file_type = $file_type_data[0] ?? NULL;
              $error = $error_data[0] ?? NULL;
            }
            else {
              $file_name = $file_name_data;
              $tmp_name = $tmp_name_data;
              $file_type = $file_type_data;
              $error = $error_data;
            }

            if ($error === UPLOAD_ERR_OK && !empty($tmp_name) && is_string($tmp_name) && file_exists($tmp_name)) {
              $raw_file = file_get_contents($tmp_name);
              $image_files[] = new ImageFile($raw_file, $file_type, $file_name);
              $this->logger->info('Processed additional image: @name', [
                '@name' => $file_name,
              ]);
            }
          }
        }
      }

      $this->logger->info('Total image files for generation: @count', [
        '@count' => count($image_files),
      ]);

      // Create input based on number of images.
      if (count($image_files) > 1) {
        // Use MultiImageToImageInput for multiple images.
        $input = new MultiImageToImageInput($image_files[0]);
        for ($i = 1; $i < count($image_files); $i++) {
          $input->addAdditionalImage($image_files[$i]);
        }
        $this->logger->info('Using @count images for composite generation', ['@count' => count($image_files)]);
      }
      else {
        // Use standard ImageToImageInput for single image.
        $input = new ImageToImageInput($image_file);
      }

      $input->setPrompt($prompt);

      // Get the selected model.
      $model = $form_state->getValue('model') ?: 'gemini-2.5-flash-image';

      // Call the AI provider.
      $output = $provider->imageToImage($input, $model, ['nanobanana_editor']);

      // Get the first generated image.
      $images = $output->getNormalized();

      if (!empty($images)) {
        $generated_image = $images[0]->getBinary();
        $this->logger->info('Generated image size: @size bytes', ['@size' => strlen($generated_image)]);

        // Store the generated image in form state.
        $form_state->set('generated_image', $generated_image);
        $token = $form_state->get('generated_image_token') ?: $this->uuid->generate();
        $form_state->set('generated_image_token', $token);
        $this->tempStore->set($token, $generated_image);

        // Rebuild the form to render preview and Save button.
        $form_state->setRebuild(TRUE);
        $form_state->disableCache();
        $rebuilt_form = $this->formBuilder->rebuildForm($this->getFormId(), $form_state);
        $response->addCommand(new HtmlCommand('#nanobanana-editor-wrapper', $rebuilt_form));
      }
      else {
        throw new \Exception('No image was generated.');
      }
    }
    catch (\Exception $e) {
      // Log the error.
      $this->logger->error('Error generating image: @message', ['@message' => $e->getMessage()]);

      // Show error message.
      $error_message = $e->getMessage();

      // Provide more helpful guidance if the API returned NO_IMAGE.
      if (strpos($error_message, 'NO_IMAGE') !== FALSE) {
        $error_message = $this->t('The AI did not generate an image. Try being more specific in your prompt. For example: "Make the background blue" or "Add a sunset effect" or "Convert to a pencil sketch style".');
      }

      $response->addCommand(new HtmlCommand('#nanobanana-messages', '<div class="messages messages--error">' . $error_message . '</div>'));
    }

    return $response;
  }

  /**
   * Submit handler for the Save button.
   */
  public function saveCallback(array &$form, FormStateInterface $form_state) {
    /** @var \Drupal\media\MediaInterface $media */
    $media = $form_state->get('media');
    $generated_image = $form_state->get('generated_image');
    if (!$generated_image) {
      $token = $form_state->getValue('generated_image_token') ?? $form_state->get('generated_image_token');
      if ($token) {
        $generated_image = $this->tempStore->get($token);
      }
    }

    if (!$generated_image) {
      $this->messenger()->addError($this->t('No generated image to save.'));
      return;
    }

    try {
      // Get the source field and original image file.
      $source_field = $media->getSource()->getConfiguration()['source_field'];
      /** @var \Drupal\file\FileInterface $old_file */
      $old_file = $media->get($source_field)->entity;

      // Create a new file with the generated image.
      $directory = dirname($old_file->getFileUri());
      $this->fileSystem->prepareDirectory($directory, FileSystemInterface::CREATE_DIRECTORY);

      // Generate a unique filename.
      $filename = $this->fileSystem->basename($old_file->getFileUri());
      $destination = $directory . '/' . $filename;

      // Save the generated image.
      $new_file_uri = $this->fileSystem->saveData($generated_image, $destination, FileExists::Replace);

      if ($new_file_uri) {
        // If we replaced the file, we need to clear the image cache.
        if ($old_file->getFileUri() === $new_file_uri) {
          // Clear image style derivatives.
          image_path_flush($new_file_uri);
        }

        // Update the media entity timestamp to trigger cache invalidation.
        $media->setChangedTime($this->dateTime->getRequestTime());
        $media->save();

        $token = $form_state->getValue('generated_image_token') ?? $form_state->get('generated_image_token');
        if ($token) {
          $this->tempStore->delete($token);
        }

        $this->messenger()->addStatus($this->t('Image has been saved successfully.'));
        $form_state->setRedirect('entity.media.edit_form', ['media' => $media->id()]);
      }
      else {
        throw new \Exception('Failed to save the generated image.');
      }
    }
    catch (\Exception $e) {
      $this->messenger()->addError($this->t('Error saving image: @message', ['@message' => $e->getMessage()]));
    }
  }

  /**
   * Submit handler for the Cancel button.
   */
  public function cancelCallback(array &$form, FormStateInterface $form_state) {
    /** @var \Drupal\media\MediaInterface $media */
    $media = $form_state->get('media');
    $form_state->setRedirect('entity.media.edit_form', ['media' => $media->id()]);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    // This is handled by specific submit handlers.
  }

}
