<?php

namespace Drupal\nanobanana_editor\Form;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Form\ConfirmFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Url;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Confirmation form for deleting a NanoBanana style.
 */
final class StyleDeleteForm extends ConfirmFormBase {

  /**
   * Config settings.
   */
  const CONFIG_NAME = 'nanobanana_editor.settings';

  /**
   * The config factory service.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * The style index to delete.
   *
   * @var int
   */
  protected $styleIndex;

  /**
   * The style name.
   *
   * @var string
   */
  protected $styleName;

  /**
   * Constructs a new StyleDeleteForm.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory service.
   */
  public function __construct(ConfigFactoryInterface $config_factory) {
    $this->configFactory = $config_factory;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new self(
      $container->get('config.factory')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'nanobanana_editor_style_delete_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state, $style_index = NULL) {
    $config = $this->configFactory->get(static::CONFIG_NAME);
    $styles = $config->get('styles') ?? [];

    if ($style_index === NULL || !isset($styles[$style_index])) {
      $this->messenger()->addError($this->t('Style not found.'));
      return $this->redirect('nanobanana_editor.settings');
    }

    $this->styleIndex = $style_index;
    $this->styleName = $styles[$style_index]['name'];

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function getQuestion() {
    return $this->t('Are you sure you want to delete the style "@name"?', ['@name' => $this->styleName]);
  }

  /**
   * {@inheritdoc}
   */
  public function getCancelUrl() {
    return new Url('nanobanana_editor.settings');
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $config = $this->configFactory->getEditable(static::CONFIG_NAME);
    $styles = $config->get('styles') ?? [];

    // Remove the style.
    unset($styles[$this->styleIndex]);

    // Re-index array.
    $styles = array_values($styles);

    // Save.
    $config->set('styles', $styles)->save();

    $this->messenger()->addStatus($this->t('Style "@name" has been deleted.', ['@name' => $this->styleName]));

    // Redirect back to settings page.
    $form_state->setRedirectUrl($this->getCancelUrl());
  }

}
