<?php

namespace Drupal\nanobanana_editor\Form;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Url;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Form for adding/editing a NanoBanana style.
 */
final class StyleForm extends FormBase {

  /**
   * Config settings.
   */
  const CONFIG_NAME = 'nanobanana_editor.settings';

  /**
   * The config factory service.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * Constructs a new StyleForm.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory service.
   */
  public function __construct(ConfigFactoryInterface $config_factory) {
    $this->configFactory = $config_factory;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new self(
      $container->get('config.factory')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'nanobanana_editor_style_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state, $style_index = NULL) {
    $config = $this->configFactory->get(static::CONFIG_NAME);
    $styles = $config->get('styles') ?? [];

    // Store the index in form state.
    $form_state->set('style_index', $style_index);

    // If editing, load existing style data.
    if ($style_index !== NULL && isset($styles[$style_index])) {
      $style = $styles[$style_index];
      $form['#title'] = $this->t('Edit Style: @name', ['@name' => $style['name']]);
    }
    else {
      $style = ['name' => '', 'prompt' => ''];
      $form['#title'] = $this->t('Add Style');
    }

    $form['name'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Style Name'),
      '#description' => $this->t('Enter a descriptive name for this style (e.g., "Watercolor", "Photorealistic").'),
      '#default_value' => $style['name'],
      '#required' => TRUE,
      '#maxlength' => 100,
    ];

    $form['prompt'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Style Prompt'),
      '#description' => $this->t('Enter the prompt text that will be appended to user prompts when this style is selected.'),
      '#default_value' => $style['prompt'],
      '#required' => TRUE,
      '#maxlength' => 500,
    ];

    $form['actions'] = [
      '#type' => 'actions',
    ];

    $form['actions']['submit'] = [
      '#type' => 'submit',
      '#value' => $style_index !== NULL ? $this->t('Update style') : $this->t('Add style'),
      '#button_type' => 'primary',
    ];

    $form['actions']['cancel'] = [
      '#type' => 'link',
      '#title' => $this->t('Cancel'),
      '#url' => Url::fromRoute('nanobanana_editor.settings'),
      '#attributes' => ['class' => ['button']],
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $config = $this->configFactory->getEditable(static::CONFIG_NAME);
    $styles = $config->get('styles') ?? [];
    $style_index = $form_state->get('style_index');

    $new_style = [
      'name' => $form_state->getValue('name'),
      'prompt' => $form_state->getValue('prompt'),
    ];

    if ($style_index !== NULL) {
      // Update existing style.
      $styles[$style_index] = $new_style;
      $this->messenger()->addStatus($this->t('Style "@name" has been updated.', ['@name' => $new_style['name']]));
    }
    else {
      // Add new style.
      $styles[] = $new_style;
      $this->messenger()->addStatus($this->t('Style "@name" has been added.', ['@name' => $new_style['name']]));
    }

    // Save the updated styles.
    $config->set('styles', $styles)->save();

    // Redirect back to settings page.
    $form_state->setRedirect('nanobanana_editor.settings');
  }

}
