<?php

namespace Drupal\navigation_extra\Plugin\Menu;

use Drupal\Component\Plugin\Exception\InvalidPluginDefinitionException;
use Drupal\Component\Plugin\Exception\PluginNotFoundException;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Menu\MenuLinkDefault;
use Drupal\Core\Menu\StaticMenuLinkOverridesInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides a default implementation for menu link plugins.
 */
class TranslatedMenuLink extends MenuLinkDefault implements ContainerFactoryPluginInterface {

  /**
   * The entity represented in the menu link.
   *
   * @var \Drupal\Core\Entity\EntityInterface|null
   */
  protected ?EntityInterface $entity = NULL;

  /**
   * Constructs a new MenuLinkEntity.
   *
   * @param array $configuration
   *   A configuration array containing information about the plugin instance.
   * @param string $plugin_id
   *   The plugin_id for the plugin instance.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\Core\Menu\StaticMenuLinkOverridesInterface $static_override
   *   The static override storage.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   */
  public function __construct(
    array $configuration,
    $plugin_id,
    $plugin_definition,
    StaticMenuLinkOverridesInterface $static_override,
    EntityTypeManagerInterface $entity_type_manager,
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition, $static_override);
    try {
      $this->entity = $entity_type_manager
        ->getStorage($this->pluginDefinition['metadata']['entity_type'])
        ->load($this->pluginDefinition['metadata']['entity_id']);
    }
    catch (InvalidPluginDefinitionException | PluginNotFoundException $e) {
      \Drupal::logger('navigation_extra')->warning('TranslatedMenuLink failed to load entity: @message', [
        '@message' => $e->getMessage(),
      ]);
      $this->entity = NULL;
    }
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('menu_link.static.overrides'),
      $container->get('entity_type.manager')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getTitle() {
    if ($this->entity) {
      return (string) $this->entity->label();
    }
    return $this->pluginDefinition['title'] ?? $this->t('Missing');
  }

  /**
   * {@inheritdoc}
   */
  public function getDescription() {
    if ($this->entity && method_exists($this->entity, 'getDescription')) {
      return $this->entity->getDescription();
    }
    return parent::getDescription();
  }

  /**
   * {@inheritdoc}
   */
  public function getCacheContexts(): array {
    return $this->entity ? $this->entity->getCacheContexts() : parent::getCacheContexts();
  }

  /**
   * {@inheritdoc}
   */
  public function getCacheTags(): array {
    return $this->entity ? $this->entity->getCacheTags() : parent::getCacheTags();
  }

  /**
   * {@inheritdoc}
   */
  public function getCacheMaxAge(): int {
    return $this->entity ? $this->entity->getCacheMaxAge() : parent::getCacheMaxAge();
  }

}
