<?php

namespace Drupal\navigation_extra\Plugin\Navigation\Extra;

use Drupal\Core\Form\FormStateInterface;
use Drupal\navigation_extra\NavigationExtraPluginBase;

/**
 * An NavigationExtraPlugin for forms navigation links.
 *
 * @NavigationExtraPlugin(
 *   id = "forms",
 *   name = @Translation("Forms"),
 *   description = @Translation("Provides navigation links for forms."),
 *   weight = 4
 * )
 */
class FormsPlugin extends NavigationExtraPluginBase {

  /**
   * {@inheritdoc}
   */
  public function buildConfigForm(array &$form, FormStateInterface $form_state): array {
    $elements = parent::buildConfigForm($form, $form_state);

    if ($this->moduleHandler->moduleExists('webform')) {
      $elements['webform'] = [
        '#type' => 'details',
        '#title' => $this->t('Webform'),
      ];

      $elements['webform']['enable'] = [
        '#type' => 'checkbox',
        '#title' => $this->t('Add webform links'),
        '#description' => $this->t('Show links to webforms.'),
        '#default_value' => $this->config->get('plugins.forms.webform.enable') ?? 0,
      ];

      // How should the webform links behave?
      $elements['webform']['link_to_results'] = [
        '#type' => 'checkbox',
        '#title' => $this->t('Link to webform results'),
        '#description' => $this->t('Menu items will link to submissions if enabled, otherwise items will point to settings.'),
        '#default_value' => $this->config->get('plugins.forms.webform.link_to_results') ?? 0,
        '#states' => [
          'visible' => [
            ':input[name="forms[webform][enable]"]' => ['checked' => TRUE],
          ],
        ],
      ];
    }

    if ($this->moduleHandler->moduleExists('contact')) {
      $elements['contact'] = [
        '#type' => 'details',
        '#title' => $this->t('Contact'),
      ];

      $elements['contact']['enable'] = [
        '#type' => 'checkbox',
        '#title' => $this->t('Add contact form links'),
        '#description' => $this->t('Show links to contact forms.'),
        '#default_value' => $this->config->get('plugins.forms.contact.enable') ?? 0,
      ];
    }

    return $elements;
  }

  /**
   * {@inheritdoc}
   */
  public function alterDiscoveredMenuLinks(array &$links): void {

    $webform_enabled = ($this->moduleHandler->moduleExists('webform') && $this->config->get('plugins.forms.webform.enable') ?? 0);
    $contact_enabled = ($this->moduleHandler->moduleExists('contact') && $this->config->get('plugins.forms.contact.enable') ?? 0);
    $webform_parent = 'navigation.forms';
    $contact_parent = 'navigation.forms';

    $webform_overview_route_name = 'entity.webform.collection';
    $contact_overview_route_name = 'entity.contact_form.collection';

    // We will need to decide what the root element is going to point to:
    // Webforms take precedence over core contact, since it is more used in
    // general.
    if ($webform_enabled) {
      // Link to the webforms overview of results overview.
      $link_to_results = ($this->config->get('plugins.forms.webform.link_to_results') ?? 0);

      // Results have precedence over settings.
      if ($link_to_results) {
        $webform_overview_route_name = 'entity.webform_submission.collection';
      }

      $root_element_route_name = $webform_overview_route_name;
    }
    elseif ($contact_enabled) {
      // Link to the contact forms overview.
      $root_element_route_name = $contact_overview_route_name;
    }
    else {
      // None of the modules that provide forms are enabled so bail out.
      return;
    }

    // Create the root element if it does not exist.
    $this->addLink('navigation.forms', [
      'route_name' => $root_element_route_name,
      'title' => $this->t('Forms'),
      'weight' => $this->config->get("plugins.forms.weight") ?? 0,
      'options' => [
        'icon' => [
          'pack_id' => 'navigation_extra',
          'icon_id' => 'forms',
          'settings' => [
            'class' => 'toolbar-button__icon',
            'size' => 25,
          ],
        ],
        'attributes' => [
          'class' => [
            'navigation-extra--forms',
          ],
        ],
      ],
    ] + ($links['navigation.forms'] ?? []), $links);

    // Create the grouped items if needed.
    if ($webform_enabled && $contact_enabled) {
      // Add a group for webforms and contact if both modules are enabled.
      $this->addLink('navigation.forms.webform', [
        'route_name' => $webform_overview_route_name,
        'title' => $this->t('Webform'),
        'weight' => 0,
        'parent' => $webform_parent,
        'options' => [
          'attributes' => [
            'class' => [
              'navigation-extra--forms--webform',
            ],
          ],
        ],
      ], $links);

      // Update the parent for the item links.
      $webform_parent .= '.webform';

      // Add a group for contact forms.
      $this->addLink('navigation.forms.contact', [
        'route_name' => $contact_overview_route_name,
        'title' => $this->t('Contact'),
        'weight' => 0,
        'parent' => $contact_parent,
        'options' => [
          'attributes' => [
            'class' => [
              'navigation-extra--forms--contact',
            ],
          ],
        ],
      ], $links);

      // Update the parent for the item links.
      $contact_parent .= '.contact';
    }

    // Add the item links for webforms.
    if ($webform_enabled) {

      // Create the collections, they point to the overview page.
      $this->addCollectionLinks($webform_parent, fn($collection) => ([
        'route_name' => $webform_overview_route_name,
        'route_parameters' => [
          'collection' => $collection['id'],
        ],
      ]), $links);

      $link_to_results = ($this->config->get('plugins.forms.webform.link_to_results') ?? 0);

      // Submission routes take the form id as a parameter.
      if ($link_to_results) {
        $callback = fn($item) => ([
          'route_name' => 'entity.webform.results_submissions',
          'route_parameters' => [
            'webform' => $item->id(),
          ],
        ]);
      }
      else {
        $callback = fn($item) => ([
          'route_name' => 'entity.webform.settings',
          'route_parameters' => [
            'webform' => $item->id(),
          ],
        ]);
      }

      $this->addItemLinks($webform_parent, 'webform', $callback, $links);
    }

    // Add the links to the contact forms.
    if ($contact_enabled) {

      // Create the collections, they point to the overview page.
      $this->addCollectionLinks($contact_parent, fn($collection) => ([
        'route_name' => $contact_overview_route_name,
        'route_parameters' => [
          'collection' => $collection['id'],
        ],
      ]), $links);

      // Contact forms don't store submisions so link to the settings always.
      $this->addItemLinks($contact_parent, 'contact_form', fn($item) => ([
        'route_name' => 'entity.contact_form.edit_form',
      ]), $links);
    }

  }

}
