<?php

declare(strict_types=1);

namespace Drupal\navigation_extra\Form;

use Drupal\Component\Plugin\Exception\PluginException;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Menu\MenuLinkManagerInterface;
use Drupal\navigation_extra\NavigationExtraPluginManagerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Settings form for navigation extra.
 *
 * @package Drupal\navigation_extra\Form
 */
class SettingsForm extends ConfigFormBase {

  /**
   * Constructs a navigation extra SettingsForm object.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $configFactory
   *   The factory for configuration objects.
   * @param \Drupal\Core\Config\TypedConfigManagerInterface|null $typedConfigManager
   *   The typed config manager.
   * @param \Drupal\Core\Extension\ModuleHandlerInterface $moduleHandler
   *   The module handler service.
   * @param \Drupal\navigation_extra\NavigationExtraPluginManagerInterface $pluginManager
   *   The navigation extra plugin manager.
   * @param \Drupal\Core\Menu\MenuLinkManagerInterface $menuLinkManager
   *   The menu link manager.
   */
  public function __construct(
    ConfigFactoryInterface $configFactory,
    $typedConfigManager,
    protected ModuleHandlerInterface $moduleHandler,
    protected NavigationExtraPluginManagerInterface $pluginManager,
    protected MenuLinkManagerInterface $menuLinkManager,
  ) {
    parent::__construct($configFactory, $typedConfigManager);
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('config.factory'),
      $container->get('config.typed'),
      $container->get('module_handler'),
      $container->get('navigation_extra.manager'),
      $container->get('plugin.manager.menu.link')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'navigation_extra_settings';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames(): array {
    return [
      'navigation_extra.settings',
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    $form = parent::buildForm($form, $form_state);

    $form['#attached']['library'][] = 'core/drupal.vertical-tabs';

    $form['tabs'] = [
      '#type' => 'vertical_tabs',
    ];

    $plugins = $this->pluginManager->getDefinitions();
    foreach ($plugins as $plugin_id => $definition) {
      try {
        /** @var \Drupal\navigation_extra\NavigationExtraPluginInterface $plugin */
        $plugin = $this->pluginManager->createInstance($plugin_id, $definition);

        // Plugin tab.
        $form['tabs'][$plugin_id] = [
          '#type' => 'details',
          '#title' => $definition['name'],
          '#group' => 'tabs',
          '#tree' => TRUE,
        ];

        // Plugin specific settings.
        $form['tabs'][$plugin_id] = $plugin->buildConfigForm($form, $form_state) + (array) $form['tabs'][$plugin_id];

        foreach ($form['tabs'][$plugin_id] as $key => &$field) {
          if ($key[0] === '#' || $key === 'enabled' || $field['#type'] == 'hidden' || $plugin_id == 'common') {
            continue;
          }

          // Automatically hide all config for a plugin if it is not enabled.
          $states = $field['#states'] ?? [];

          if (empty($states['visible'])) {
            $states['visible'] = [
              [
                ':input[name="' . $plugin_id . '[enabled]"]' => ['checked' => TRUE],
              ],
            ];
          }
          else {
            $states['visible'][0][':input[name="' . $plugin_id . '[enabled]"]'] = ['checked' => TRUE];
          }

          $field['#states'] = $states;

        }

      }
      catch (PluginException) {
        // Silently continue if plugin is not found.
      }
    }

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {

    // Prepare values.
    $plugins = $this->pluginManager->getDefinitions();
    foreach ($plugins as $plugin_id => $definition) {
      /** @var \Drupal\navigation_extra\NavigationExtraPluginInterface $plugin */
      try {
        $plugin = $this->pluginManager->createInstance($plugin_id, $definition);
        $plugin->submitConfigForm($form, $form_state);
      }
      catch (PluginException) {
        // Continue silently.
      }
    }

    $form_state->addCleanValueKey('tabs__active_tab');
    $form_state->cleanValues();

    $values = $form_state->getValues();

    // Move submitted plugin values into the 'plugins' key.
    $values['plugins'] = [];
    $plugins = $this->pluginManager->getDefinitions();
    foreach ($plugins as $plugin_id => $definition) {
      $values['plugins'][$plugin_id] = $values[$plugin_id];
      unset($values[$plugin_id]);
    }

    // Save the config.
    $config = $this->config('navigation_extra.settings');
    $config->setData($values);
    $config->save();

    // Clear cache so admin menu can rebuild.
    $this->menuLinkManager->rebuild();

    parent::submitForm($form, $form_state);
  }

}
