<?php

declare(strict_types=1);

namespace Drupal\navigation_extra;

use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Form\FormStateInterface;

/**
 * Provides an interface defining a MenuLinksPlugin.
 */
interface NavigationExtraPluginInterface {

  /**
   * Check if this plugin is enabled or not.
   *
   * @return bool
   *   Whether this plugin is enabled.
   */
  public function isEnabled(): bool;

  /**
   * Alter discovered menu links before the normal alter hook.
   *
   * @param array $links
   *   The links array.
   */
  public function preAlterDiscoveredMenuLinks(array &$links): void;

  /**
   * Alter discovered menu links.
   *
   * Allows a plugin to hook into the alteration of existing discovered menu
   * items.
   *
   * @param array $links
   *   The links array.
   */
  public function alterDiscoveredMenuLinks(array &$links): void;

  /**
   * Alter discovered menu links after the normal alter hook.
   *
   * @param array $links
   *   The links array.
   */
  public function postAlterDiscoveredMenuLinks(array &$links): void;

  /**
   * Given an entity, determine if a menu rebuild needs to be triggered.
   *
   * @param \Drupal\Core\Entity\EntityInterface $entity
   *   The entity.
   *
   * @return bool
   *   Whether the menu needs to rebuild.
   */
  public function needsMenuLinkRebuild(EntityInterface $entity): bool;

  /**
   * Allows a plugin to hook into the menu preprocess.
   *
   * @param array $variables
   *   The variables array.
   */
  public function preprocessMenu(array &$variables): void;

  /**
   * Allows a plugin to hook into page attachments.
   *
   * @param array $page
   *   The page array.
   */
  public function pageAttachments(array &$page): void;

  /**
   * Build a config form.
   *
   * Allows a plugin to add extra settings to the admin toolbar content
   * configuration settings form.
   *
   * @param array $form
   *   The form.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   *
   * @return array
   *   The form array.
   */
  public function buildConfigForm(array &$form, FormStateInterface $form_state): array;

  /**
   * Submit handler for config form.
   *
   * Allows a plugin to handle stuff during submission of the config form.
   *
   * @param array $form
   *   The form.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   */
  public function submitConfigForm(array &$form, FormStateInterface $form_state): void;

}
