<?php

declare(strict_types=1);

namespace Drupal\navigation_extra;

use Drupal\Component\Plugin\Exception\PluginException;
use Drupal\Component\Plugin\Exception\PluginNotFoundException;
use Drupal\Core\Cache\CacheBackendInterface;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Menu\MenuLinkManagerInterface;
use Drupal\Core\Plugin\DefaultPluginManager;
use Drupal\Core\State\StateInterface;

/**
 * Manages navigation extra plugins.
 */
class NavigationExtraPluginManager extends DefaultPluginManager implements NavigationExtraPluginManagerInterface {

  /**
   * A Drupal State object.
   *
   * @var \Drupal\Core\State\StateInterface
   */
  protected StateInterface $state;

  /**
   * A Menu link plugin manager object.
   *
   * @var \Drupal\Core\Menu\MenuLinkManagerInterface
   */
  protected MenuLinkManagerInterface $menuLinkPluginManager;

  /**
   * Constructs a new NavigationExtraPluginManager object.
   *
   * @param \Traversable $namespaces
   *   An object that implements \Traversable which contains the root paths
   *   keyed by the corresponding namespace to look for plugin implementations.
   * @param \Drupal\Core\Cache\CacheBackendInterface $cache_backend
   *   An object that implements CacheBackendInterface.
   * @param \Drupal\Core\Extension\ModuleHandlerInterface $module_handler
   *   An object that implements ModuleHandlerInterface.
   * @param \Drupal\Core\State\StateInterface $state
   *   A Drupal State object.
   * @param \Drupal\Core\Menu\MenuLinkManagerInterface $menu_link_plugin_manager
   *   A Menu link plugin manager object.
   */
  public function __construct(
    \Traversable $namespaces,
    CacheBackendInterface $cache_backend,
    ModuleHandlerInterface $module_handler,
    StateInterface $state,
    MenuLinkManagerInterface $menu_link_plugin_manager,
  ) {
    parent::__construct(
      'Plugin/Navigation/Extra',
      $namespaces,
      $module_handler,
      'Drupal\navigation_extra\NavigationExtraPluginInterface',
      'Drupal\navigation_extra\Annotation\NavigationExtraPlugin'
    );
    $this->setCacheBackend($cache_backend, 'navigation_extra_plugins');
    $this->alterInfo('navigation_extra_plugins_info');
    $this->state = $state;
    $this->menuLinkPluginManager = $menu_link_plugin_manager;
  }

  /**
   * {@inheritdoc}
   */
  protected function findDefinitions() {
    $definitions = parent::findDefinitions();
    // If this plugin depends on a module that does not exist, remove the
    // plugin definition.
    foreach ($definitions as $plugin_id => $plugin_definition) {
      $providers = ((array) $plugin_definition)['dependencies'] ?? [];
      foreach ($providers as $provider) {
        if (!$this->providerExists($provider)) {
          unset($definitions[$plugin_id]);
          break;
        }
      }
    }
    return $definitions;
  }


  /**
   * {@inheritdoc}
   */
  public function menuLinkRebuild(EntityInterface $entity): void {

    if ($this->state->get('system.maintenance_mode')) {
      // We only re-generate the menu if we are not in maintenance mode.
      return;
    }

    $needsRebuild = FALSE;

    $plugins = $this->getDefinitions();

    foreach ($plugins as $plugin_id => $definition) {
      try {
        $plugin = $this->createInstance($plugin_id, $definition);
        if ($plugin->isEnabled() && $plugin->needsMenuLinkRebuild($entity)) {
          $needsRebuild = TRUE;
          break;
        }
      }
      catch (PluginException) {
        // Silently continue if plugin is not found.
      }
    }

    if ($needsRebuild) {
      $this->menuLinkPluginManager->rebuild();
    }
  }

  /**
   * {@inheritdoc}
   */
  public function menuLinksDiscoveredAlter(array &$links): void {

    $plugins = $this->getDefinitions();

    // Allows plugins to run code before any other plugin.
    foreach ($plugins as $plugin_id => $definition) {
      try {
        $plugin = $this->createInstance($plugin_id, $definition);
        if ($plugin->isEnabled()) {
          $plugin->preAlterDiscoveredMenuLinks($links);
        }
      }
      catch (PluginException) {
        // Silently continue if plugin is not found.
      }
    }

    // Run normal discovery.
    foreach ($plugins as $plugin_id => $definition) {
      try {
        $plugin = $this->createInstance($plugin_id, $definition);
        if ($plugin->isEnabled()) {
          $plugin->alterDiscoveredMenuLinks($links);
        }
      }
      catch (PluginException) {
        // Silently continue if plugin is not found.
      }
    }

    // Allows plugins to run code after the other plugins.
    foreach ($plugins as $plugin_id => $definition) {
      try {
        $plugin = $this->createInstance($plugin_id, $definition);
        if ($plugin->isEnabled()) {
          $plugin->postAlterDiscoveredMenuLinks($links);
        }
      }
      catch (PluginException) {
        // Silently continue if plugin is not found.
      }
    }
  }

  /**
   * {@inheritdoc}
   */
  public function preprocessMenu(array &$variables): void {

    $plugins = $this->getDefinitions();

    foreach ($plugins as $plugin_id => $definition) {
      try {
        $plugin = $this->createInstance($plugin_id, $definition);
        if ($plugin->isEnabled()) {
          $plugin->preprocessMenu($variables);
        }
      }
      catch (PluginException) {
        // Silently continue if plugin is not found.
      }
    }
  }

  /**
   * {@inheritdoc}
   */
  public function pageAttachments(array &$page): void {

    $plugins = $this->getDefinitions();

    foreach ($plugins as $plugin_id => $definition) {
      try {
        $plugin = $this->createInstance($plugin_id, $definition);
        if ($plugin->isEnabled()) {
          $plugin->pageAttachments($page);
        }
      }
      catch (PluginException) {
        // Silently continue if plugin is not found.
      }
    }
  }

  /**
   * {@inheritdoc}
   */
  public function isEnabled(string $plugin_id): bool {
    try {
      $definition = $this->getDefinition($plugin_id);
      $plugin = $this->createInstance($plugin_id, $definition);
      return $plugin->isEnabled();
    }
    catch (PluginNotFoundException | PluginException) {
      return FALSE;
    }
  }

}
