<?php

namespace Drupal\navigation_extra\Plugin\Navigation\Extra;

use Drupal\Core\Form\FormStateInterface;
use Drupal\navigation_extra\NavigationExtraPluginBase;

/**
 * An NavigationExtraPlugin for navigation links.
 *
 * @NavigationExtraPlugin(
 *   id = "common",
 *   name = @Translation("Common"),
 *   description = @Translation("Common settings for navigation extra."),
 *   weight = 0
 * )
 */
class CommonPlugin extends NavigationExtraPluginBase {

  const NAVIGATION_MAX_DEPTH = 50;

  /**
   * {@inheritdoc}
   */
  public function buildConfigForm(array &$form, FormStateInterface $form_state): array {
    $elements = parent::buildConfigForm($form, $form_state);

    // Always enabled.
    $elements['enabled']['#type'] = 'hidden';
    $elements['enabled']['#value'] = 1;

    // Weight doesn't matter for common settings.
    $elements['weight']['#type'] = 'hidden';
    $elements['weight']['#value'] = 0;

    // Show common as the first tab.
    $elements['#weight'] = -99;

    $elements['group_collections'] = [
      '#type' => 'radios',
      '#title' => $this->t("Group collections"),
      '#description' => $this->t("Group collection items together to either the top or bottom of the menu tree."),
      '#options' => [
        '' => $this->t("Don't Group"),
        'bottom' => $this->t("At bottom"),
        'top' => $this->t("At top"),
      ],
      '#default_value' => $this->config->get('plugins.common.group_collections') ?? '',
    ];

    $elements['hide_empty_collections'] = [
      '#type' => 'checkbox',
      '#title' => $this->t("Hide empty collections"),
      '#description' => $this->t("Hide empty collection menu items."),
      '#default_value' => $this->config->get('plugins.common.hide_empty_collections') ?? 0,
    ];

    $elements['hide_add_new_content'] = [
      '#type' => 'checkbox',
      '#title' => $this->t("Hide add new content"),
      '#description' => $this->t("Hide the add new content item."),
      '#default_value' => $this->config->get('plugins.common.hide_add_new_content') ?? 0,
    ];

    $elements['generate_overview_links'] = [
      '#type' => 'checkbox',
      '#title' => $this->t("Generate overview links"),
      '#description' => $this->t("Generates an overview link if a menu item has children beneath so the item itself can still be reached."),
      '#default_value' => $this->config->get('plugins.common.generate_overview_links') ?? 0,
    ];

    $elements['override_max_menu_depth'] = [
      '#type' => 'checkbox',
      '#title' => $this->t("Override max menu depth"),
      '#description' => $this->t("Core is limited to 3 levels, with this option you can override that behavior."),
      '#default_value' => $this->config->get('plugins.common.override_max_menu_depth') ?? 0,
    ];

    return $elements;
  }

  /**
   * {@inheritdoc}
   */
  public function alterDiscoveredMenuLinks(array &$links): void {
    $hide_core_link = $this->config->get("plugins.common.hide_add_new_content") ?? FALSE;

    if ($hide_core_link) {
      $this->removeLink('navigation.create', $links);
    }

  }

  /**
   * {@inheritdoc}
   */
  public function preprocessMenu(array &$variables): void {
    parent::preprocessMenu($variables);

    $generate_overview_links = $this->config->get('plugins.common.generate_overview_links') ?? FALSE;
    if ($generate_overview_links) {
      $items = $variables['items'] ?? [];
      $this->generateOverviewItems($items);
      $variables['items'] = $items;
    }
  }

  /**
   * Generate a overview menu item links.
   *
   * @param array $items
   *   An array of menu items.
   * @param int $level
   *   The current recursive level begin processed.
   */
  protected function generateOverviewItems(array &$items, int $level = 0): void {
    foreach ($items as $menu_id => &$item) {
      if (!empty($item['below'])) {
        // Generate the "Overview" item.
        if ($level > 0) {
          $overview_item = $this->generateOverviewItem($item);

          $item['below'] = [
            $menu_id => $overview_item,
          ] + $item['below'];

        }

        // Recursively call this function for the child items.
        $this->generateOverviewItems($item['below'], $level + 1);
      }
    }
  }

  /**
   * Generate a single overview menu item link.
   *
   * @param array $item
   *   A menu item begin processed.
   *
   * @return array
   *   The menu item altered into an overview item.
   */
  protected function generateOverviewItem(array $item): array {
    $item['below'] = [];
    $item['title'] = $this->t('Overview');
    return $item;
  }

  /**
   * {@inheritdoc}
   */
  public function pageAttachments(&$page): void {
    $page['#attached']['library'][] = 'navigation_extra/navigation_extra';
  }

}
