<?php

namespace Drupal\navigation_extra\Plugin\Navigation\Extra;

use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\navigation_extra\NavigationExtraPluginBase;

/**
 * An NavigationExtraPlugin for content navigation links.
 *
 * @NavigationExtraPlugin(
 *   id = "content",
 *   name = @Translation("Content"),
 *   description = @Translation("Provides navigation links for content."),
 *   weight = 1,
 *   entity_type = "node_type",
 *   dependencies = {
 *     "node"
 *   }
 * )
 */
class ContentPlugin extends NavigationExtraPluginBase {

  /**
   * {@inheritdoc}
   */
  public function buildConfigForm(array &$form, FormStateInterface $form_state): array {
    $elements = parent::buildConfigForm($form, $form_state);

    $elements += $this->buildConfigFormRecentItems($form, $form_state);
    $elements += $this->buildConfigFormCreateItems($form, $form_state);

    return $elements;
  }

  /**
   * Add config fields to a plugin config form when it is using recent items.
   *
   * @param array $form
   *   The complete config form.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   *
   * @return array
   *   An array of elements for configuring the recent items.
   */
  protected function buildConfigFormRecentItems(array &$form, FormStateInterface $form_state): array {

    $elements = [];

    $elements['recent_items'] = [
      '#type' => 'details',
      '#title' => $this->t('Recent content'),
    ];

    $elements['recent_items']['number_of_items'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Number of items'),
      '#description' => $this->t('The amount of items to show. Set to 0 or leave empty to disable recent items.'),
      '#default_value' => $this->config->get('plugins.content.recent_items.number_of_items') ?? 5,
    ];

    $elements['recent_items']['hide_empty_list'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Hide empty list'),
      '#description' => $this->t('Hide recent content list if non are available.'),
      '#default_value' => $this->config->get('plugins.content.recent_items.hide_empty_list') ?? FALSE,
    ];

    $recent_items_link_options = [
      'default' => $this->t('Edit form'),
      'view' => $this->t('View'),
    ];

    // Add Layout Builder option only if layout builder is enabled.
    if ($this->moduleHandler->moduleExists('layout_builder')) {
      $recent_items_link_options['layout_builder'] = $this->t('Layout Builder');
    }

    $elements['recent_items']['link'] = [
      '#type' => 'radios',
      '#title' => $this->t('Link'),
      '#description' => $this->t('Choose where recent items should be linked to.'),
      '#options' => $recent_items_link_options,
      '#default_value' => $this->config->get('plugins.content.link') ?? 'default',
    ];

    return $elements;
  }

  /**
   * {@inheritdoc}
   */
  public function alterDiscoveredMenuLinks(array &$links): void {

    // Create the root element if it does not exist.
    $this->addLink('navigation.content', [
      'route_name' => 'view.content.page_1',
      'title' => $this->t('Content'),
      'weight' => $this->config->get("plugins.content.weight") ?? 0,
      'options' => [
        'attributes' => [
          'class' => [
            'navigation-extra--content',
          ],
        ],
      ],
    ] + ($links['navigation.content'] ?? []), $links);

    // Create the collections.
    $this->addCollectionLinks(
      'navigation.content',
      fn($collection) => ([
        'route_name' => 'view.content.page_1',
        'route_parameters' => [
          'collection' => $collection['id'],
        ],
      ]),
      $links
    );

    $this->addItemLinks(
      'navigation.content',
      'node_type',
      fn($item) => ([
        'route_name' => 'view.content.page_1',
        'route_parameters' => [
          'type' => $item->id(),
        ],
      ]),
      $links
    );

    $this->addCreateEntityLinks(
      'navigation.content',
      'node.add_page',
      'node.add',
      'node_type',
      $links
    );

    // $this->createItemRecentContentLinks('node');
  }

  /**
   * {@inheritdoc}
   */
  public function needsMenuLinkRebuild(EntityInterface $entity): bool {
    return (bool) ($this->config->get('plugins.content.recent_items.number_of_items') ?? 0);
  }

}
