<?php

declare(strict_types=1);

namespace Drupal\navigation_extra\Plugin\Block;

use Drupal\Core\Access\AccessResult;
use Drupal\Core\Access\AccessResultInterface;
use Drupal\Core\Block\Attribute\Block;
use Drupal\Core\Menu\LocalTaskManagerInterface;
use Drupal\Core\Menu\Plugin\Block\LocalTasksBlock;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Routing\RouteMatchInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\navigation_extra\NavigationExtraPluginManagerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Defines a local tasks navigation block class.
 */
#[Block(
  id: 'navigation_extra_local_tasks',
  admin_label: new TranslatableMarkup('Local Tasks'),
)]
class NavigationExtraLocalTasksBlock extends LocalTasksBlock implements ContainerFactoryPluginInterface {

  /**
   * Constructs a new NavigationLocalTasksBlock.
   *
   * @param array $configuration
   *   A configuration array containing information about the plugin instance.
   * @param string $plugin_id
   *   The plugin ID for the plugin instance.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\Core\Menu\LocalTaskManagerInterface $local_task_manager
   *   The local task manager.
   * @param \Drupal\Core\Routing\RouteMatchInterface $route_match
   *   The route match.
   * @param \Drupal\navigation_extra\NavigationExtraPluginManagerInterface $navigationExtra
   *   The navigation extra plugin handler service.
   */
  public function __construct(
    array $configuration,
    $plugin_id,
    $plugin_definition,
    LocalTaskManagerInterface $local_task_manager,
    RouteMatchInterface $route_match,
    protected NavigationExtraPluginManagerInterface $navigationExtra,
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition, $local_task_manager, $route_match);
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('plugin.manager.menu.local_task'),
      $container->get('current_route_match'),
      $container->get('navigation_extra.manager'),
    );
  }

  /**
   * {@inheritdoc}
   */
  protected function blockAccess(AccountInterface $account): AccessResultInterface {
    return AccessResult::allowedIf($this->navigationExtra->isEnabled('local_tasks'));
  }

  /**
   * {@inheritdoc}
   */
  public function build(): array {
    $build = parent::build();

    // Transform the build tabs into usable navigation links.
    if (empty($build['#primary'])) {
      return [
        '#cache' => $build['#cache'],
      ];
    }

    // Sort the links by their weight.
    uasort($build['#primary'], [
      'Drupal\Component\Utility\SortArray',
      'sortByWeightProperty',
    ]);

    $local_task_links = [];
    foreach ($build['#primary'] as $primary_link) {

      if (($primary_link['#access'] instanceof AccessResult && !$primary_link['#access']->isAllowed()) || !$primary_link['#access']) {
        continue;
      }

      // Get only the link part of the tab.
      $local_task_link = $primary_link['#link'];

      // On the active tab, put the secondary level below if available.
      if ($primary_link['#active'] && !empty($build['#secondary'])) {
        uasort($build['#secondary'], [
          'Drupal\Component\Utility\SortArray',
          'sortByWeightProperty',
        ]);

        foreach ($build['#secondary'] as $secondary_link) {
          if (($secondary_link['#access'] instanceof AccessResult && !$secondary_link['#access']->isAllowed()) || !$secondary_link['#access']) {
            continue;
          }
          $local_task_link['below'][] = $secondary_link['#link'];
        }
      }

      // @todo Add extra attributes to the link.
      $local_task_links[] = $local_task_link;
    }

    $label = $this->configuration['label'] ?? $this->t('Local Tasks');

    $local_task_items = [
      [
        'title' => $label,
        'class' => 'navigation-local-tasks',
        'below' => $local_task_links,
      ],
    ];

    return [
      '#theme' => 'navigation_menu',
      '#menu_name' => 'local_tasks',
      '#title' => $label,
      '#items' => $local_task_items,
      '#cache' => $build['#cache'],
    ];
  }

}
