<?php

namespace Drupal\navigation_extra\Plugin\Navigation\Extra;

use Drupal\Core\Form\FormStateInterface;
use Drupal\navigation_extra\NavigationExtraPluginBase;

/**
 * An NavigationExtraPlugin for media navigation links.
 *
 * @NavigationExtraPlugin(
 *   id = "blocks",
 *   name = @Translation("Blocks"),
 *   description = @Translation("Provides navigation links for blocks."),
 *   weight = 0,
 *   entity_type = "block_content",
 *   dependencies = {
 *     "block"
 *   }
 * )
 */
class BlocksPlugin extends NavigationExtraPluginBase {

  /**
   * {@inheritdoc}
   */
  public function buildConfigForm(array &$form, FormStateInterface $form_state): array {
    $elements = parent::buildConfigForm($form, $form_state);

    $elements['hide_core_link'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Hide core link'),
      '#description' => $this->t('Hide link provided by core.'),
      '#default_value' => $this->config->get("plugins.blocks.hide_core_link") ?? FALSE,
    ];

    $elements['position'] = [
      '#type' => 'select',
      '#options' => [
        ''  => $this->t('Top level (default)'),
        'navigation.content' => $this->t('Content'),
      ],
      '#title' => $this->t('Position'),
      '#description' => $this->t('Place the block content links under a different menu item.'),
      '#default_value' => $this->config->get("plugins.blocks.position") ?? '',
      '#states' => [
        'invisible' => [
          [
            ':input[name="blocks[hide_core_link]"]' => ['checked' => TRUE],
          ],
        ],
      ],
    ];

    $elements['navigation_safe_block_ids'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Navigation safe block ids'),
      '#description' => $this->t('Allow these blocks to be used in navigation. Use a newline to separate block ids.'),
      '#default_value' => $this->config->get("plugins.blocks.navigation_safe_block_ids") ?? FALSE,
    ];

    $elements['hidden_navigation_safe_block_ids'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Hidden navigation safe blocks ids'),
      '#description' => $this->t('Hide these blocks in the blocks admin ui (structure). Use a newline to separate block ids.'),
      '#default_value' => $this->config->get("plugins.blocks.hidden_navigation_safe_block_ids") ?? FALSE,
    ];

    return $elements;
  }

  /**
   * {@inheritdoc}
   */
  public function alterDiscoveredMenuLinks(array &$links): void {

    $hide_core_link = $this->config->get("plugins.blocks.hide_core_link") ?? 0;

    if ($hide_core_link) {
      $this->removeLink('navigation.blocks', $links);
    }
    else {
      $position = $this->config->get('plugins.blocks.position') ?? '';

      // Create the root element if it does not exist.
      $link = [
          'route_name' => 'view.block_content.page_1',
          'title' => $this->t('Blocks'),
          'weight' => $this->config->get("plugins.blocks.weight") ?? 0,
          'parent' => $position ?: '',
          'options' => [
            'attributes' => [
              'class' => [
                'navigation-extra--blocks',
              ],
            ],
          ],
        ] + ($links['navigation.blocks'] ?? []);

      $this->addLink('navigation.blocks', $link, $links);
    }

  }

}
