<?php

namespace Drupal\navigation_extra\Plugin\Navigation\Extra;

use Drupal\Core\Form\FormStateInterface;
use Drupal\navigation_extra\NavigationExtraPluginBase;

/**
 * An NavigationExtraPlugin for media navigation links.
 *
 * @NavigationExtraPlugin(
 *   id = "media",
 *   name = @Translation("Media"),
 *   description = @Translation("Provides navigation links for media."),
 *   weight = 3,
 *   entity_type = "media",
 *   dependencies = {
 *     "media"
 *   }
 * )
 */
class MediaPlugin extends NavigationExtraPluginBase {

  /**
   * {@inheritdoc}
   */
  public function buildConfigForm(array &$form, FormStateInterface $form_state): array {
    $elements = parent::buildConfigForm($form, $form_state);

    if ($this->isRouteAvailable('view.media_library.page')) {
      $elements['link_media_library'] = [
        '#type' => 'checkbox',
        '#title' => $this->t('Link to media library'),
        '#description' => $this->t('Links media items directly to the media library.'),
        '#default_value' => $this->config->get("plugins." . $this->getPluginId() . '.link_media_library') ?? 0,
      ];
    }

    $elements += $this->buildConfigFormCreateItems($form, $form_state);

    return $elements;
  }

  /**
   * {@inheritdoc}
   */
  public function alterDiscoveredMenuLinks(array &$links): void {
    $link_library = $this->config->get('plugins.media.link_media_library') ?? TRUE;

    // Use Media Library page if library is accessible.
    if ($link_library && $this->isRouteAvailable('view.media_library.page')) {
      $route_name = 'view.media_library.page';
    }
    else {
      $route_name = 'view.media.media_page_list';
    }

    $this->addLink('navigation.media', [
      'route_name' => $route_name,
      'title' => $this->t('Media'),
      'weight' => $this->config->get("plugins.media.weight") ?? 0,
      'options' => [
        'attributes' => [
          'class' => [
            'navigation-extra--media',
          ],
        ],
      ],
    ] + ($links['navigation.media'] ?? []), $links);

    // Create the collections.
    $this->addCollectionLinks(
      'navigation.media',
      fn($collection) => ([
        'route_name' => $route_name,
        'route_parameters' => [
          'collection' => $collection['id'],
        ],
      ]),
      $links
    );

    $this->addItemLinks(
      'navigation.media',
      'media_type',
      fn($item) => ([
        'route_name' => $route_name,
        'route_parameters' => [
          'type' => $item->id(),
        ],
      ]),
      $links
    );

    $this->addCreateEntityLinks(
      'navigation.media',
      $route_name,
      'entity.media.add_form',
      'media_type',
      $links
    );

  }

}
