<?php

namespace Drupal\navigation_extra\Plugin\Navigation\Extra;

use Drupal\Core\Form\FormStateInterface;
use Drupal\navigation_extra\NavigationExtraPluginBase;

/**
 * Manages development and administration tool links.
 *
 * @NavigationExtraPlugin(
 *   id = "tools",
 *   name = @Translation("Tools"),
 *   description = @Translation("Manages development and administration tool links."),
 *   weight = 0
 * )
 */
class ToolsPlugin extends NavigationExtraPluginBase {

  /**
   * {@inheritdoc}
   */
  public function buildConfigForm(array &$form, FormStateInterface $form_state): array {
    $elements = parent::buildConfigForm($form, $form_state);

    // Weight doesn't matter for common settings.
    $elements['weight']['#type'] = 'hidden';
    $elements['weight']['#value'] = 0;

    // If navigation_extra_tools is installed add some options to move it under
    // version.
    if ($this->moduleHandler->moduleExists('navigation_extra_tools')) {
      $elements['navigation_extra_tools'] = [
        '#type' => 'details',
        '#title' => $this->t('Navigation extra tools'),
      ];

      $elements['navigation_extra_tools']['position'] = [
        '#type' => 'select',
        '#options' => [
          '' => $this->t('Tools menu (default)'),
          'navigation_extra_version' => $this->t('Version menu'),
          'user' => $this->t('User menu'),
        ],
        '#title' => $this->t('Position'),
        '#description' => $this->t('Place the tools links under a different menu item.'),
        '#default_value' => $this->config->get("plugins.tools.navigation_extra_tools.position") ?? '',
      ];

      $elements['navigation_extra_tools']['group'] = [
        '#type' => 'checkbox',
        '#title' => $this->t('Group under Tools'),
        '#description' => $this->t('Keep all links under a Tools item.'),
        '#default_value' => $this->config->get('plugins.tools.navigation_extra_tools.group') ?? 0,
        '#states' => [
          'visible' => [
            ':input[name="tools[navigation_extra_tools][position]"]' => ['value' => 'navigation_extra_version'],
          ],
        ],
      ];
    }

    if ($this->moduleHandler->moduleExists('devel')) {
      $elements['devel'] = [
        '#type' => 'details',
        '#title' => $this->t('Devel'),
      ];

      $elements['devel']['position'] = [
        '#type' => 'select',
        '#options' => [
          '' => $this->t('Devel menu (default)'),
          'navigation_extra_version' => $this->t('Version menu'),
          'admin' => $this->t('Administration menu'),
        ],
        '#title' => $this->t('Position'),
        '#description' => $this->t('Place the devel links under a different menu item.'),
        '#default_value' => $this->config->get("plugins.tools.devel.position") ?? '',
      ];

      $elements['devel']['group'] = [
        '#type' => 'checkbox',
        '#title' => $this->t('Group under Development'),
        '#description' => $this->t('Keep all links under a Development item.'),
        '#default_value' => $this->config->get('plugins.tools.devel.group') ?? 0,
        '#states' => [
          'visible' => [
            ':input[name="tools[devel][position]"]' => ['value' => 'navigation_extra_version'],
          ],
        ],
      ];
    }

    return $elements;
  }

  /**
   * {@inheritdoc}
   */
  public function alterDiscoveredMenuLinks(array &$links): void {

    if ($this->moduleHandler->moduleExists('navigation_extra_tools')) {
      $this->alterDiscoveredMenuLinksTools($links);
    }

    if ($this->moduleHandler->moduleExists('devel')) {
      $this->alterDiscoveredMenuLinksDevel($links);
    }
  }

  /**
   * Alter discovered menu links for Navigation Extra Tools.
   *
   * @param array $links
   *   The links array.
   */
  public function alterDiscoveredMenuLinksTools(array &$links): void {
    $position = $this->config->get('plugins.tools.navigation_extra_tools.position') ?? '';
    $group = $this->config->get('plugins.tools.navigation_extra_tools.group') ?? 0;

    if ($position == 'navigation_extra_version' && $this->config->get('plugins.version.enabled')) {
      if ($group) {
        // Just change the parent of the main item.
        $links['navigation_extra_tools.help']['parent'] = 'navigation.version';
        foreach ($links as $id => &$link) {
          if (str_contains($id, 'navigation_extra_tools')) {
            if ('admin' == ($link['menu_name'] ?? '')) {
              $link['menu_name'] = 'navigation_extra_version';
            }
          }
        }
      }
      else {
        // Move all items to the new parent and remove the original.
        foreach ($links as $id => &$link) {
          if (str_contains($id, 'navigation_extra_tools')) {
            if ($link['parent'] == 'navigation_extra_tools.help') {
              $link['parent'] = 'navigation.version';
            }
          }
        }

        $this->removeLink('navigation_extra_tools.help', $links);
      }
    }
  }

  /**
   * Alter discovered menu links for Devel.
   *
   * @param array $links
   *   The links array.
   */
  public function alterDiscoveredMenuLinksDevel(array &$links): void {

    $position = $this->config->get('plugins.tools.devel.position') ?? '';
    $group = $this->config->get('plugins.tools.devel.group') ?? 0;
    $menu = 'devel';
    $parent = NULL;
    $item_parent = NULL;

    switch ($position) {
      case 'navigation_extra_version':
        if (!$this->config->get('plugins.version.enabled')) {
          // Bail out, version is not enabled, so revert to default behavior.
          break;
        }

        $menu = 'navigation_extra_version';
        $parent = 'navigation.version';

        break;

      case 'admin':
        // Move to the admin menu and force group under "Development".
        $menu = 'admin';
        $parent = 'system.admin';
        $group = TRUE;

        break;

      default:
        // Do nothing.
        return;
    }


    if ($group) {
      // Generate a parent item.
      $this->addLink('navigation.tools.development', [
        'route_name' => 'devel.admin_settings',
        'title' => $this->t('Development'),
        'weight' => $this->config->get("plugins.tools.weight") ?? 0,
        'menu_name' => $menu,
        'parent' => $parent,
        'options' => [
          'attributes' => [
            'class' => [
              'navigation-extra--tools',
              'navigation-extra--tools-development',
            ],
          ],
        ],
      ] + ($links['navigation.tools.development'] ?? []), $links);

      $item_parent = 'navigation.tools.development';
    }

    // Move all items of devel to the new parent and menu.
    foreach ($links as $id => &$link) {
      if (str_starts_with($id, 'devel')) {
        if ('devel' == ($link['menu_name'] ?? '')) {
          $link['menu_name'] = $menu;
          $link['parent'] = $item_parent ?: $parent;
        }
      }
    }
  }

}
