<?php

declare(strict_types=1);

namespace Drupal\navigation_menu_role\Plugin\Block;

use Drupal\Core\Access\AccessResult;
use Drupal\Core\Access\AccessResultInterface;
use Drupal\Core\Block\Attribute\Block;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Menu\MenuActiveTrailInterface;
use Drupal\Core\Menu\MenuLinkTreeInterface;
use Drupal\Core\Menu\MenuTreeParameters;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\navigation\Plugin\Derivative\SystemMenuNavigationBlock as SystemMenuNavigationBlockDeriver;
use Drupal\system\Plugin\Block\SystemMenuBlock;
use Drupal\user\RoleInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides a generic menu navigation block whose visibility is managed by role.
 *
 * This class should extend NavigationMenuBlock, but it is declared as final
 * while the module is experimental. Hopefully that bit could be removed during
 * the stabilization phase.
 *
 * @see \Drupal\navigation\Plugin\Block\NavigationMenuBlock
 */
#[Block(
  id: "navigation_menu_role",
  admin_label: new TranslatableMarkup("Navigation menu (role visibility)"),
  category: new TranslatableMarkup("Menus per role (Navigation)"),
  deriver: SystemMenuNavigationBlockDeriver::class,
)]
final class NavigationMenuRoleBlock extends SystemMenuBlock implements ContainerFactoryPluginInterface {

  const NAVIGATION_MAX_DEPTH = 3;

  /**
   * Constructs a new NavigationMenuRoleBlock.
   *
   * @param array $configuration
   *   A configuration array containing information about the plugin instance.
   * @param string $plugin_id
   *   The plugin ID for the plugin instance.
   * @param array $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\Core\Menu\MenuLinkTreeInterface $menu_tree
   *   The menu tree service.
   * @param \Drupal\Core\Menu\MenuActiveTrailInterface $menu_active_trail
   *   The active menu trail service.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   The entity type manager.
   */
  public function __construct(
    array $configuration,
    $plugin_id,
    $plugin_definition,
    MenuLinkTreeInterface $menu_tree,
    MenuActiveTrailInterface $menu_active_trail,
    protected EntityTypeManagerInterface $entityTypeManager,
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition, $menu_tree, $menu_active_trail);
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition): static {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('navigation.menu_tree'),
      $container->get('menu.active_trail'),
      $container->get('entity_type.manager'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration(): array {
    return [
      'level' => 1,
      'depth' => 0,
      'roles' => [],
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function blockForm($form, FormStateInterface $form_state): array {
    $form = parent::blockForm($form, $form_state);
    unset($form['menu_levels']['expand_all_items']);
    $form['menu_levels']['depth']['#options'] = range(1, static::NAVIGATION_MAX_DEPTH);

    $roles = $this->entityTypeManager->getStorage('user_role')->loadMultiple();
    $role_options = array_map(function (RoleInterface $role) {
      return $role->label();
    }, $roles);

    $config = $this->configuration;
    $defaults = $this->defaultConfiguration();

    $form['roles_wrapper'] = [
      '#type' => 'details',
      '#title' => $this->t('Roles'),
      // Open if not set to defaults.
      '#open' => $defaults['roles'] !== $config['roles'],
      '#process' => [[self::class, 'processMenuLevelParents']],
    ];

    // Checkbox form element.
    $form['roles_wrapper']['roles'] = [
      '#type' => 'checkboxes',
      '#title' => $this->t('Select roles'),
      '#options' => $role_options,
      '#default_value' => $this->configuration['roles'],
      '#description' => $this->t('Select the roles allowed to view this block. Leave all the options unchecked to grant access to all roles.'),
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function blockSubmit($form, FormStateInterface $form_state): void {
    $this->configuration['level'] = $form_state->getValue('level');
    $this->configuration['depth'] = $form_state->getValue('depth');
    $this->configuration['roles'] = array_values(array_filter($form_state->getValue('roles')));
  }

  /**
   * {@inheritdoc}
   */
  public function build(): array {
    $menu_name = $this->getDerivativeId();
    $level = $this->configuration['level'];
    $depth = $this->configuration['depth'];
    $parameters = new MenuTreeParameters();
    $parameters
      ->setMinDepth($level)
      ->setMaxDepth(min($level + $depth, $this->menuTree->maxDepth()))
      ->onlyEnabledLinks();
    $tree = $this->menuTree->load($menu_name, $parameters);
    $manipulators = [
      ['callable' => 'menu.default_tree_manipulators:checkAccess'],
      ['callable' => 'menu.default_tree_manipulators:generateIndexAndSort'],
    ];
    $tree = $this->menuTree->transform($tree, $manipulators);
    $build = $this->menuTree->build($tree);
    if (!empty($build)) {
      $build['#title'] = $this->configuration['label'];
    }

    return $build;
  }

  /**
   * {@inheritdoc}
   */
  protected function blockAccess(AccountInterface $account): AccessResultInterface {
    return AccessResult::allowedIf(empty($this->configuration['roles']) || !empty(array_intersect($account->getRoles(), $this->configuration['roles'])));
  }

  /**
   * {@inheritdoc}
   */
  public function calculateDependencies() {
    return [
      'module' => [
        'system',
      ],
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function getCacheContexts(): array {
    // We don't use menu active trails here.
    return array_filter(parent::getCacheContexts(), static fn (string $tag) => !str_starts_with($tag, 'route.menu_active_trails'));
  }

}
