<?php

declare(strict_types=1);

namespace Drupal\navigation_plus;

use Drupal\Core\Entity\EntityInterface;
use Symfony\Component\HttpFoundation\RequestStack;
use Symfony\Component\HttpKernel\Event\ResponseEvent;

/**
 * Flag new page.
 *
 * The page builder UI can be configured to do things on the initial page edit.
 * For instance if you create a Layout Builder controlled note, you can drop
 * into Edit mode right away.
 *
 * @see Drupal.NavigationPlus.contentWasJustCreated().
 */
final class FlagNewPage {

  /**
   * Constructs a FlagNewPage object.
   */
  public function __construct(
    private readonly NavigationPlusUi $navigationPlusUi,
    private readonly RequestStack $requestStack,
  ) {}

  /**
   * Flag new entity.
   *
   * @param \Drupal\Core\Entity\EntityInterface $entity
   *  An entity that was just created.
   *
   * @return void
   */
  public function flagNewEntity(EntityInterface $entity): void {
    $session = $this->requestStack->getCurrentRequest()->getSession();
    $just_created = $session->get('just_created', []);
    $just_created[$entity->bundle()][$entity->id()] = $entity->id();
    $session->set('just_created', $just_created);
  }


  /**
   * Inform front end of new entity.
   *
   * @param ResponseEvent $event
   *   The kernel response.
   *
   * @return void
   */
  public function informFrontEndOfNewEntity(ResponseEvent &$event) {
    $entity = $this->navigationPlusUi->deriveEntityFromRoute();
    if (!empty($entity)) {
      $session = $this->requestStack->getCurrentRequest()->getSession();
      $just_created = $session->get('just_created', []);
      $response = $event->getResponse();
      if (!empty($just_created[$entity->bundle()][$entity->id()]) && str_contains($response->headers->get('Content-Type'), 'html')) {
        $settings_js = <<<EOD
<script type="text/javascript">
  var NavigationPlusEntityJustCreated = true;
</script>
EOD;
        $new_content = str_replace('</body>', $settings_js . '</body>', $response->getContent());
        $response->setContent($new_content);
        unset($just_created[$entity->bundle()][$entity->id()]);
        $session->set('just_created', $just_created);
      }
    }
  }

}
