<?php

namespace Drupal\navigation_plus;

use Drupal\Core\Ajax\AjaxResponse;
use Drupal\Core\Ajax\InvokeCommand;
use Drupal\Core\Ajax\MessageCommand;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Render\RendererInterface;
use Drupal\navigation_plus\Ajax\ClearMessagesCommand;
use Drupal\navigation_plus\Ajax\EditModeMessageCommand;

/**
 * Provides common form handling methods for Navigation Plus.
 */
trait NavigationPlusFormTrait {

  /**
   * Form AJAX submit callback.
   */
  public function updateAjaxSubmit(array &$form, FormStateInterface $form_state) {
    $response = new AjaxResponse();

    if ($form_state->getTemporaryValue('updatePage')) {
      $this->updatePage($response, $form, $form_state);
    }
    $input = $form_state->getUserInput();
    if (
      $form_state->getTemporaryValue('updateForm') ||
      !empty($input['previously_had_errors']) ||
      !empty($input['settings']['block_form']['empty_field'])
    ) {
      $this->removeMessages($response);
      $this->updateForm($response, $form, $form_state);
    }

    $this->renderMessages($response);
    $this->handleErrors($response, $form, $form_state);

    $response->addCommand(new InvokeCommand(NULL, 'editPlusIsDoneUpdating'));
    return $response;
  }

  /**
   * Render messages.
   *
   * @param \Drupal\Core\Ajax\AjaxResponse $response
   *   The AJAX response object.
   */
  public function renderMessages(AjaxResponse $response, $use_edit_mode_message = TRUE) {
    $messenger = \Drupal::messenger();
    $messages = $messenger->all();

    if (!empty($messages)) {
      // Clear the messenger service since we're handling display ourselves.
      $messenger->deleteAll();

      $first_message = TRUE;
      foreach ($messages as $type => $type_messages) {
        foreach ($type_messages as $message) {
          if ($use_edit_mode_message) {
            $response->addCommand(new EditModeMessageCommand($message, $type));
          } else {
            $response->addCommand(new MessageCommand(
              $message,
              NULL,
              ['type' => $type],
              $first_message   // Only clear previous messages on the first message.
            ));
            $first_message = FALSE;
          }
        }
      }
    }
  }

  /**
   * Remove messages from the screen.
   *
   * @param \Drupal\Core\Ajax\AjaxResponse $response
   *   The AJAX response object.
   */
  public function removeMessages(AjaxResponse $response) {
    $response->addCommand(new ClearMessagesCommand());
  }

  /**
   * Gets the renderer service.
   *
   * @return \Drupal\Core\Render\RendererInterface
   *   The renderer service.
   */
  protected function getRenderer(): RendererInterface {
    return \Drupal::service('renderer');
  }

}
