<?php

declare(strict_types=1);

namespace Drupal\navigation_plus\Message;

use Drupal\Core\EventSubscriber\MainContentViewSubscriber;
use Drupal\Core\Security\TrustedCallbackInterface;

/**
 * Handles status message rendering for Navigation Plus Edit Mode.
 *
 * When in Edit Mode, messages are intercepted by the EditModeMessenger
 * decorator. This service attaches them as drupalSettings during render
 * for client-side delivery via initial-messages.js.
 */
final class EditModeHtmlMessages implements TrustedCallbackInterface {

  public function __construct(
    private readonly EditModeMessenger $messenger,
  ) {}

  /**
   * {@inheritdoc}
   */
  public static function trustedCallbacks() {
    return ['preRender', 'renderMessages'];
  }

  /**
   * Pre-render callback for status_messages elements.
   *
   * When in Edit Mode, this uses a custom lazy builder that stores messages
   * in drupalSettings for client-side rendering. Otherwise, it uses the
   * default Drupal status messages rendering.
   *
   * @param array $element
   *   A renderable array.
   *
   * @return array
   *   The updated renderable array containing the placeholder.
   */
  public static function preRender(array $element): array {
    $navigationPlusUi = \Drupal::service('navigation_plus.ui');
    $request = \Drupal::request();

    // Detect when content is rendered during an AJAX request using Drupal's
    // wrapper format convention. For AJAX requests, use default lazy builder so
    // EditModeAjaxMessages handles delivery.
    $wrapper_format = $request->query->get(MainContentViewSubscriber::WRAPPER_FORMAT, $request->getRequestFormat());
    $ajax_formats = ['drupal_ajax', 'drupal_modal', 'drupal_dialog'];
    $is_ajax = in_array($wrapper_format, $ajax_formats, TRUE);

    // Determine which lazy builder to use based on whether we're in Edit Mode.
    if ($navigationPlusUi->getMode() === 'edit' && !$is_ajax) {
      // Use our custom lazy builder for Edit Mode.
      $lazy_builder = [
        'navigation_plus.messages.html:renderMessages',
        [$element['#display']],
      ];
    }
    else {
      // Use the default Drupal lazy builder.
      $lazy_builder = [
        'Drupal\Core\Render\Element\StatusMessages::renderMessages',
        [$element['#display']],
      ];
    }

    $build = [
      '#lazy_builder' => $lazy_builder,
      '#create_placeholder' => TRUE,
    ];

    // Directly create a placeholder as we need this to be placeholdered
    // regardless if this is a POST or GET request.
    $build = \Drupal::service('render_placeholder_generator')->createPlaceholder($build);

    if ($element['#include_fallback']) {
      return [
        'fallback' => [
          '#markup' => '<div data-drupal-messages-fallback class="hidden"></div>',
        ],
        'messages' => $build,
      ];
    }
    return $build;
  }

  /**
   * Lazy builder callback for rendering status messages.
   *
   * In Edit Mode, gets stored messages from EditModeMessenger decorator
   * and attaches them as drupalSettings for client-side delivery.
   * Otherwise, uses default Drupal rendering.
   *
   * @return array
   *   A renderable array.
   */
  public function renderMessages(): array {
    $stored_messages = $this->messenger->getStoredMessages();

    if (empty($stored_messages)) {
      return [];
    }

    $this->messenger->clearStoredMessages();

    // Attach messages to drupalSettings for initial-messages.js.
    return [
      '#attached' => [
        'drupalSettings' => [
          'navigationPlus' => [
            'initialMessages' => $stored_messages,
          ],
        ],
        'library' => [
          'navigation_plus/edit_mode',
        ],
      ],
    ];
  }

}
