<?php

namespace Drupal\neuraflow\Form;

use Drupal\Core\Condition\ConditionPluginCollection;
use Drupal\Core\Executable\ExecutableManagerInterface;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Form\SubformState;
use Drupal\Core\Language\LanguageManagerInterface;
use Drupal\Core\Plugin\Context\ContextRepositoryInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * The setting form for the neurabot settings.
 *
 * @package Drupal\neuraflow\Form
 */
class NeurabotConfigForm extends ConfigFormBase {

  const CONFIG_OBJECT_NAME = 'neuraflow.settings';

  /**
   * The condition plugin manager.
   *
   * @var \Drupal\Core\Executable\ExecutableManagerInterface
   */
  protected ExecutableManagerInterface $conditionPluginManager;

  /**
   * The visibility collection.
   *
   * @var \Drupal\Core\Condition\ConditionPluginCollection
   */
  protected ConditionPluginCollection $visibilityCollection;

  /**
   * The context manager service.
   *
   * @var \Drupal\Core\Plugin\Context\ContextRepositoryInterface
   */
  protected ContextRepositoryInterface $contextRepository;

  /**
   * The language manager service.
   *
   * @var \Drupal\Core\Language\LanguageManagerInterface
   */
  protected LanguageManagerInterface $languageManager;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    $instance = parent::create($container);

    $instance->conditionPluginManager = $container->get('plugin.manager.condition');
    $instance->contextRepository = $container->get('context.repository');
    $instance->languageManager = $container->get('language_manager');

    return $instance;
  }

  /**
   * Gets conditions for this block.
   *
   * @return \Drupal\Core\Condition\ConditionPluginCollection
   *   An array or collection of configured condition plugins.
   */
  public function getVisibilityConditions(): ConditionPluginCollection {
    if (!isset($this->visibilityCollection)) {
      $this->visibilityCollection = new ConditionPluginCollection($this->conditionPluginManager, $this->config(self::CONFIG_OBJECT_NAME)->get('neurabot.visibility') ?? []);
    }
    return $this->visibilityCollection;
  }

  /**
   * Returns an array of visibility condition configurations.
   *
   * @return array
   *   An array of visibility condition configuration keyed by the condition ID.
   */
  public function getVisibility(): array {
    return $this->getVisibilityConditions()->getConfiguration();
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'neurabot_config_form';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames(): array {
    return [
      'neuraflow.settings',
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    // Store the gathered contexts in the form state for other objects to use
    // during form building.
    $form_state->setTemporaryValue('gathered_contexts', $this->contextRepository->getAvailableContexts());

    $form['#tree'] = TRUE;
    $form['general'] = [
      '#type' => 'details',
      '#title' => $this->t('General settings'),
      '#open' => TRUE,
    ];

    $form['general']['assistantId'] = [
      '#title' => $this->t('Assistant ID'),
      '#description' => $this->t('Enter your Assistant ID'),
      '#type' => 'textfield',
      '#config_target' => self::CONFIG_OBJECT_NAME . ':neurabot.chatLinkConfig.assistantId',
      '#required' => TRUE,
    ];

    $form['general']['environment'] = [
      '#title' => $this->t('Environment'),
      '#description' => $this->t('Enter your environment name'),
      '#type' => 'select',
      '#options' => [
        'DEVELOPMENT' => $this->t('Development'),
        'STAGING' => $this->t('Staging'),
        'PRODUCTION' => $this->t('Production'),
      ],
      '#config_target' => self::CONFIG_OBJECT_NAME . ':neurabot.chatLinkConfig.environment',
      '#required' => TRUE,
    ];

    $form['general']['mode'] = [
      '#title' => $this->t('Mode'),
      '#description' => $this->t('Enter your mode'),
      '#type' => 'select',
      '#options' => [
        'FLOATING' => $this->t('Floating'),
        'EMBEDDED' => $this->t('Embedded'),
        'FULLSCREEN' => $this->t('Fullscreen'),
      ],
      '#config_target' => self::CONFIG_OBJECT_NAME . ':neurabot.chatLinkConfig.mode',
      '#required' => TRUE,
    ];

    $form['general']['containerId'] = [
      '#title' => $this->t('Container ID'),
      '#description' => $this->t('Enter your container ID where the chatbot should be appended to.'),
      '#type' => 'textfield',
      '#config_target' => self::CONFIG_OBJECT_NAME . ':neurabot.chatLinkConfig.containerId',
      '#states' => [
        'visible' => [
          ':input[name="general[mode]"]' => ['value' => 'EMBEDDED'],
        ],
      ],
    ];

    $form['general']['outsideMessages'] = [
      '#title' => $this->t('Outside messages'),
      '#description' => $this->t('Display outside messages'),
      '#type' => 'checkbox',
      '#config_target' => self::CONFIG_OBJECT_NAME . ':neurabot.chatLinkConfig.outsideMessages',
    ];

    $form['general']['useIsolatedFrames'] = [
      '#title' => $this->t('Use isolated frames'),
      '#description' => $this->t('Use isolated frames'),
      '#type' => 'checkbox',
      '#config_target' => self::CONFIG_OBJECT_NAME . ':neurabot.chatLinkConfig.useIsolatedFrames',
    ];

    // Visibility settings.
    $form['visibility'] = $this->buildVisibilityInterface([], $form_state);

    return parent::buildForm($form, $form_state);
  }

  /**
   * Helper function for building the visibility UI form.
   *
   * @param array $form
   *   An associative array containing the structure of the form.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The current state of the form.
   *
   * @return array
   *   The form array with the visibility UI added in.
   */
  protected function buildVisibilityInterface(array $form, FormStateInterface $form_state): array {
    $form['visibility_tabs'] = [
      '#type' => 'vertical_tabs',
      '#title' => $this->t('Visibility'),
      '#parents' => ['visibility_tabs'],
      '#attached' => [
        'library' => [
          'neuraflow/neurabot.admin',
        ],
      ],
    ];

    $visibility = $this->getVisibility();
    $conditionDefinitions = $this->conditionPluginManager->getDefinitionsForContexts($form_state->getTemporaryValue('gathered_contexts'));
    foreach ($conditionDefinitions as $condition_id => $definition) {
      // Don't display the current theme condition.
      if ($condition_id == 'current_theme') {
        continue;
      }
      // Don't display the language condition until we have multiple languages.
      if ($condition_id == 'language' && !$this->languageManager->isMultilingual()) {
        continue;
      }

      /** @var \Drupal\Core\Condition\ConditionInterface $condition */
      $condition = $this->conditionPluginManager->createInstance($condition_id, $visibility[$condition_id] ?? []);
      $form_state->set(['conditions', $condition_id], $condition);
      $condition_form = $condition->buildConfigurationForm([], $form_state);
      $condition_form['#type'] = 'details';
      $condition_form['#title'] = $condition->getPluginDefinition()['label'];
      $condition_form['#group'] = 'visibility_tabs';
      $form[$condition_id] = $condition_form;
    }

    // Disable negation for specific conditions.
    $disable_negation = [
      'entity_bundle:node',
      'language',
      'response_status',
      'user_role',
    ];

    foreach ($disable_negation as $condition) {
      if (isset($form[$condition])) {
        $form[$condition]['negate']['#type'] = 'value';
        $form[$condition]['negate']['#value'] = $form[$condition]['negate']['#default_value'];
      }
    }

    if (isset($form['user_role'])) {
      $form['user_role']['#title'] = $this->t('Roles');
      unset($form['user_role']['roles']['#description']);
    }

    if (isset($form['request_path'])) {
      $form['request_path']['#title'] = $this->t('Pages');
      $form['request_path']['negate']['#type'] = 'radios';
      $form['request_path']['negate']['#default_value'] = (int) $form['request_path']['negate']['#default_value'];
      $form['request_path']['negate']['#title_display'] = 'invisible';
      $form['request_path']['negate']['#options'] = [
        $this->t('Show for the listed pages'),
        $this->t('Hide for the listed pages'),
      ];
    }

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state): void {
    parent::validateForm($form, $form_state);

    foreach ($form_state->getValue('visibility') as $condition_id => $values) {
      // Allow the condition to validate the form.
      /** @var \Drupal\Core\Condition\ConditionInterface $condition */
      $condition = $form_state->get(['conditions', $condition_id]);
      $condition->validateConfigurationForm($form['visibility'][$condition_id], SubformState::createForSubform($form['visibility'][$condition_id], $form, $form_state));
    }
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    parent::submitForm($form, $form_state);

    foreach ($form_state->getValue('visibility') as $condition_id => $values) {
      // Allow the condition to submit the form.
      /** @var \Drupal\Core\Condition\ConditionInterface $condition */
      $condition = $form_state->get(['conditions', $condition_id]);
      $condition->submitConfigurationForm($form['visibility'][$condition_id], SubformState::createForSubform($form['visibility'][$condition_id], $form, $form_state));

      $condition_configuration = $condition->getConfiguration();
      // Update the visibility conditions on the block.
      $this->visibilityCollection->addInstanceId($condition_id, $condition_configuration);
    }

    // Set visibility settings explicitly due to no presence of
    // #config_target concept for plugin sub forms.
    $config = $this->config(self::CONFIG_OBJECT_NAME);
    $config->set('neurabot.visibility', $this->visibilityCollection->getConfiguration());
    $config->save();
  }

}
