<?php

namespace Drupal\nexaguard_cmp\Controller;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Controller\ControllerBase;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\Response;

/**
 * Provides a standalone HTML view for NexaGuard CMP.
 *
 * This controller bypasses Drupal theming and returns a complete,
 * self-contained HTML page with CMP visuals and confetti animation.
 */
final class StandaloneController extends ControllerBase {

  /**
   * The config factory service.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  private ConfigFactoryInterface $configFactory;

  /**
   * Constructs a new StandaloneController instance.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory.
   */
  public function __construct(ConfigFactoryInterface $config_factory) {
    $this->configFactory = $config_factory;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): self {
    return new self(
      $container->get('config.factory'),
    );
  }

  /**
   * Renders the standalone NexaGuard CMP HTML page.
   *
   * @return \Symfony\Component\HttpFoundation\Response
   *   A raw HTML response bypassing Drupal theming.
   */
  public function page(): Response {
    $cfg = $this->configFactory->get('nexaguard_cmp.settings');

    $settingsId = trim((string) ($cfg->get('settings_id') ?? ''));
    $assetsUrl  = trim((string) ($cfg->get('assets_url') ?: 'https://cmp.nexaguard.com'));
    $apiUrl     = trim((string) ($cfg->get('api_url') ?: 'https://api.nexaguard.io'));
    $loaderUrl  = trim((string) ($cfg->get('loader_url') ?: 'https://cmp.nexaguard.com/loader.js'));

    // Build the CMP loader tag from config (no Drupal theming occurs here).
    $cmpTag = '';
    if ($settingsId !== '') {
      $cmpTag = sprintf(
        '<script id="nexaguard-cmp" data-settings-id="%s" data-api-url="%s" data-assets-url="%s" src="%s"></script>',
        htmlspecialchars($settingsId, ENT_QUOTES, 'UTF-8'),
        htmlspecialchars($apiUrl, ENT_QUOTES, 'UTF-8'),
        htmlspecialchars($assetsUrl, ENT_QUOTES, 'UTF-8'),
        htmlspecialchars($loaderUrl, ENT_QUOTES, 'UTF-8')
      );
    }

    // Full standalone HTML output.
    $html = <<<HTML
<!DOCTYPE html>
<html lang="en">
<head>
  <meta charset="UTF-8">
  <title>Welcome to NexaGuard™ CMP!</title>
  <link rel="icon" href="https://nexaguard.com/favicon.ico">
  <meta name="viewport" content="width=device-width, initial-scale=1.0">
  <style>
    :root{--bg1:#1d1f21;--bg2:#3a3c3d;--fg:#c5c8c6;--accent:#81a2be;--link:#8be9fd;--celebrate:#FFD700;--celebrate2:#FF69B4;--celebrate3:#90ee90;--celebrate4:#ff6347;--celebrate5:#00e1ff}
    *,*::before,*::after{box-sizing:border-box;margin:0;padding:0}
    body{min-height:100vh;font-family:'Segoe UI',Tahoma,Geneva,Verdana,sans-serif;text-align:center;padding:50px 20px;background:linear-gradient(135deg,var(--bg1),var(--bg2));color:var(--fg);line-height:1.6;overflow-y:auto}
    @keyframes float{0%,100%{transform:translateY(0)}50%{transform:translateY(-10px)}}
    img.logo{width:120px;display:block;margin:0 auto 20px;animation:float 3s ease-in-out infinite}
    h1{font-size:2em;margin-bottom:.3em;color:var(--accent)}
    .celebrate{position:relative;margin:0 auto 30px auto;padding:20px 36px 18px 36px;font-size:1.5em;font-weight:700;background:linear-gradient(90deg,var(--celebrate),var(--celebrate3),var(--celebrate2));color:#2d2d2d;border-radius:20px;box-shadow:0 2px 16px #0003;max-width:600px;letter-spacing:.03em;overflow:hidden;z-index:0}
    .celebrate small{display:block;font-size:.8em;color:#333;margin-top:3px;font-weight:400}
    .cmp-id{display:inline-block;background:#0008;color:var(--celebrate);border-radius:8px;padding:2px 13px 2px 10px;margin-left:10px;font-size:.7em;font-weight:700;letter-spacing:.1em;vertical-align:middle;position:relative;top:-2px;border:2px solid var(--celebrate)}
    .confetti{pointer-events:none;position:fixed;left:0;top:0;width:100vw;height:100vh;z-index:0;overflow:visible}
    p{font-size:1.2em;margin:1em auto;max-width:700px}
    a{color:var(--link);text-decoration:none;border-bottom:1px dashed var(--link)}
    a:hover{text-decoration:underline}
    .footer{margin-top:50px;font-size:.9em;color:#6272a4}
    .reload{color:var(--celebrate2);font-size:1.05em;margin-top:18px;font-weight:500;display:inline-block;opacity:.85}
    .celebrate-emoji{font-size:1.8em;vertical-align:-3px;margin-right:8px;animation:bounce .9s infinite alternate}
    @keyframes bounce{0%{transform:translateY(-3px)}100%{transform:translateY(8px)}}
  </style>

  <!-- Google Analytics + Consent Mode (optional here; CMP will set consent) -->
  <script async src="https://www.googletagmanager.com/gtag/js?id=G-YCR1CZNLS"></script>
  <script>
    window.dataLayer = window.dataLayer || [];
    function gtag(){dataLayer.push(arguments);}
    gtag('consent', 'default', { ad_storage:'denied', analytics_storage:'denied' });
    gtag('js', new Date());
    gtag('config', 'G-YCR1CZNLS');
  </script>
</head>
<body>
  <canvas class="confetti"></canvas>
  <img src="/modules/custom/nexaguard_cmp/assets/cmplogo.svg" alt="NexaGuard Logo" class="logo">
  <img src="/modules/custom/nexaguard_cmp/assets/IABTCF.svg" alt="TCF Registered CMP" style="width:200px;margin:0 auto 22px;display:block;">

  <div class="celebrate">
    <span class="celebrate-emoji">🎉</span>
    We did it! NexaGuard CMP is now <b>IAB TCF v2.2 Validated</b>!
    <span class="cmp-id">CMP&nbsp;ID&nbsp;471</span>
    <small>All those late nights finally paid off. Thank you to everyone who believed!</small>
  </div>

  <h1>Welcome to NexaGuard™ CMP!</h1>
  <p>Powered by caffeine, code, and a relentless chase of GDPR specs, our CMP stands guard 24/7.</p>
  <p>We wrestled with obscure vendor lists at 3 AM, survived infinite test loops, and bribed our servers with extra RAM, all to deliver enterprise-grade peace of mind.</p>
  <p>Got feedback? <a href="https://nexaguard.com/" target="_blank" rel="noopener">Drop us a line</a>, we’ll only reboot if you’re really, really sure.</p>
  <div class="footer">
    Disclaimer: No cookies were harmed in the making of this CMP.<br>
    404 errors are merely tech’s way of saying “plot twist.”
  </div>
  <div class="reload">Reload for more confetti! 🎊</div>

  <script>
    // Confetti animation.
    const confetti = document.querySelector('.confetti');
    confetti.width = window.innerWidth; confetti.height = window.innerHeight;
    const palette = ['var(--celebrate)','var(--celebrate2)','var(--celebrate3)','var(--celebrate4)','var(--celebrate5)','var(--accent)','var(--link)'];
    function rc(){ return palette[Math.floor(Math.random()*palette.length)]; }
    function draw(ctx,x,y,r,color,rot){ ctx.save(); ctx.translate(x,y); ctx.rotate(rot); ctx.beginPath(); ctx.arc(0,0,r,0,2*Math.PI); ctx.fillStyle = color; ctx.globalAlpha=0.85; ctx.fill(); ctx.restore(); }
    function make(n){ const a=[]; for(let i=0;i<n;i++){ a.push({x:Math.random()*confetti.width,y:Math.random()*confetti.height,r:5+Math.random()*10,c:rc(),s:1.2+Math.random()*2.5,ang:Math.random()*2*Math.PI,dr:(Math.random()-0.5)*0.15,rot:Math.random()*2*Math.PI,spin:(Math.random()-0.5)*0.12}); } return a;}
    let pieces = make(38 + Math.floor(Math.random()*9));
    function anim(){ const ctx=confetti.getContext('2d'); ctx.clearRect(0,0,confetti.width,confetti.height);
      for(const p of pieces){ p.y+=p.s; p.x+=Math.sin(p.ang+=p.dr)*1.5; if(p.y>confetti.height+20){ p.y=-12; p.x=Math.random()*confetti.width;} draw(ctx,p.x,p.y,p.r,p.c,p.rot+=p.spin); }
      requestAnimationFrame(anim);
    } anim();
    addEventListener('resize', ()=>{ confetti.width=innerWidth; confetti.height=innerHeight; pieces = make(pieces.length); });
  </script>

  {$cmpTag}
</body>
</html>
HTML;

    // Return raw Response (bypasses Drupal theming entirely).
    $response = new Response($html, 200, ['Content-Type' => 'text/html; charset=UTF-8']);
    // Prevent aggressive proxies from caching while iterating.
    $response->setPrivate();

    return $response;
  }

}
