<?php

namespace Drupal\nexaguard_cmp\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Url;

/**
 * Defines the settings form for the NexaGuard CMP integration.
 */
final class NexaGuardSettingsForm extends ConfigFormBase {

  /**
   * {@inheritdoc}
   *
   * Returns the unique form ID for NexaGuard CMP settings.
   */
  public function getFormId(): string {
    return 'nexaguard_cmp_settings_form';
  }

  /**
   * {@inheritdoc}
   *
   * Returns the list of editable configuration names.
   */
  protected function getEditableConfigNames(): array {
    return ['nexaguard_cmp.settings'];
  }

  /**
   * Builds the configuration form for NexaGuard CMP.
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    $cfg = $this->config('nexaguard_cmp.settings');

    // Attach admin styles.
    $form['#attached']['library'][] = 'nexaguard_cmp/admin_ui';

    // Read config.
    $settingsId = (string) ($cfg->get('settings_id') ?? '');
    $autoInject = (bool) $cfg->get('auto_inject');
    $consent    = (bool) $cfg->get('enable_gcm');
    $debug      = (bool) $cfg->get('debug_mode');
    $anonOnly   = (bool) $cfg->get('anonymous_only');
    $exAdmin    = (bool) $cfg->get('exclude_admin');

    $assetsUrl = $cfg->get('assets_url') ?: 'https://cmp.nexaguard.com';
    $apiUrl    = $cfg->get('api_url') ?: 'https://api.nexaguard.io';
    $loaderUrl = $cfg->get('loader_url') ?: 'https://cmp.nexaguard.com/loader.js';

    // Links.
    $dashboard  = Url::fromUri('https://app.nexaguard.com/')->toString();
    $homeDebug  = Url::fromUserInput('/', ['query' => ['nxg_debug' => 1]])->toString();
    $stagingUrl = Url::fromRoute('nexaguard_cmp.staging')->toString();
    $standalone = Url::fromRoute('nexaguard_cmp.standalone')->toString();

    // Ambient background.
    $form['ambient'] = [
      '#type' => 'container',
      '#attributes' => ['class' => ['ambient-bg']],
    ];

    // Wrapper containers.
    $form['nexaguard_admin'] = [
      '#type' => 'container',
      '#attributes' => ['class' => ['nexaguard-admin']],
    ];
    $form['nexaguard_admin']['container'] = [
      '#type' => 'container',
      '#attributes' => ['class' => ['nexaguard-container']],
    ];

    // --- HERO ----------------------------------------------------------------
    $form['nexaguard_admin']['container']['hero'] = [
      '#type' => 'container',
      '#attributes' => ['class' => ['nexaguard-hero']],
      'eyebrow'  => ['#markup' => '<div class="nexaguard-hero__eyebrow">Consent Management Platform</div>'],
      'title'    => ['#markup' => '<h1 class="nexaguard-hero__title">NexaGuard CMP</h1>'],
      'subtitle' => ['#markup' => '<p class="nexaguard-hero__subtitle">Configure your Drupal integration here. For analytics, vendor insights, and banner personalization, visit your Customer Dashboard.</p>'],
    ];

    // Actions row.
    $form['nexaguard_admin']['container']['hero']['actions'] = [
      '#type' => 'container',
      '#attributes' => ['class' => ['nexaguard-hero__actions']],
    ];

    // LEFT group (three compact buttons).
    $form['nexaguard_admin']['container']['hero']['actions']['left'] = [
      '#type' => 'container',
      '#attributes' => ['class' => ['nxg-actions-left']],
      'links' => [
        '#markup' =>
          '<a class="nexaguard-btn nexaguard-btn--brand" href="' . $dashboard . '" target="_blank" rel="noopener">Open Customer Dashboard</a>' .
          '<a class="nexaguard-btn nexaguard-btn--brand" href="' . $standalone . '" target="_blank" rel="noopener">View Staging</a>' .
          '<a class="nexaguard-btn nexaguard-btn--brand" href="' . $homeDebug . '" target="_blank" rel="noopener">Open Site with Debug</a>',
      ],
    ];

    // RIGHT group (Reset as link, Save as primary).
    $form['nexaguard_admin']['container']['hero']['actions']['right'] = [
      '#type' => 'container',
      '#attributes' => ['class' => ['nxg-actions-right']],
    ];

    // Reset defaults link.
    $form['nexaguard_admin']['container']['hero']['actions']['right']['reset'] = [
      '#type' => 'submit',
      '#value' => $this->t('Reset defaults'),
      '#submit' => ['::resetDefaults'],
      '#limit_validation_errors' => [],
      '#attributes' => ['class' => ['nexaguard-link-reset']],
    ];

    // Save Configuration button.
    $form['nexaguard_admin']['container']['hero']['actions']['right']['save'] = [
      '#type' => 'submit',
      '#value' => $this->t('Save Configuration'),
      '#attributes' => ['class' => ['nexaguard-btn', 'nexaguard-btn--primary']],
    ];

    // --- GRID ----------------------------------------------------------------
    $form['nexaguard_admin']['container']['grid'] = [
      '#type' => 'container',
      '#attributes' => ['class' => ['nexaguard-grid']],
    ];

    // Setup.
    $form['nexaguard_admin']['container']['grid']['setup'] = [
      '#type' => 'container',
      '#attributes' => ['class' => ['nexaguard-card']],
      'title' => ['#markup' => '<h2 class="nexaguard-card__title">Setup</h2>'],
      'desc'  => ['#markup' => '<p class="nexaguard-card__desc">Enter your App ID and toggle integrations. The loader will be auto-injected into <code>&lt;head&gt;</code>. Endpoints are fixed by NexaGuard.</p>'],
    ];

    $form['nexaguard_admin']['container']['grid']['setup']['form'] = [
      '#type' => 'container',
      '#attributes' => ['class' => ['nxg-form']],
    ];

    $form['nexaguard_admin']['container']['grid']['setup']['form']['settings_id'] = [
      '#type' => 'textfield',
      '#title' => $this->t('App ID (settings-id)'),
      '#default_value' => $settingsId,
      '#required' => TRUE,
      '#attributes' => ['placeholder' => 'NXG-XXXXXXX', 'class' => ['regular-text']],
    ];

    $form['nexaguard_admin']['container']['grid']['setup']['form']['auto_inject'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Auto-inject CMP Script'),
      '#default_value' => $autoInject,
    ];

    $form['nexaguard_admin']['container']['grid']['setup']['form']['enable_gcm'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable Google Consent Mode v2'),
      '#default_value' => $consent,
    ];

    $form['nexaguard_admin']['container']['grid']['setup']['form']['debug_mode'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Debug Mode'),
      '#default_value' => $debug,
    ];

    $form['nexaguard_admin']['container']['grid']['setup']['form']['anonymous_only'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Inject only for anonymous users'),
      '#default_value' => $anonOnly,
      '#attributes' => ['style' => 'margin-top:6px;'],
    ];

    $form['nexaguard_admin']['container']['grid']['setup']['form']['exclude_admin'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Exclude admin pages'),
      '#default_value' => $exAdmin,
    ];

    // Status section.
    $form['nexaguard_admin']['container']['grid']['status'] = [
      '#type' => 'container',
      '#attributes' => ['class' => ['nexaguard-card']],
      'title' => ['#markup' => '<h2 class="nexaguard-card__title">Status</h2>'],
      'list'  => [
        '#markup' =>
          '<ul class="nexaguard-list">' .
          '<li><span>Auto-inject</span><span class="nexaguard-badge ' . ($autoInject ? 'is-on' : 'is-off') . '">' . ($autoInject ? 'ON' : 'OFF') . '</span></li>' .
          '<li><span>Consent Mode</span><span class="nexaguard-badge ' . ($consent ? 'is-on' : 'is-off') . '">' . ($consent ? 'ON' : 'OFF') . '</span></li>' .
          '<li><span>App ID</span><code>' . $this->e($settingsId ?: '—') . '</code></li>' .
          '<li><span>Assets URL</span><code>' . $this->e($assetsUrl) . '</code></li>' .
          '<li><span>API URL</span><code>' . $this->e($apiUrl) . '</code></li>' .
          '<li><span>Loader</span><code>' . $this->e($loaderUrl) . '</code></li>' .
          '<li><span>Module Version</span><code>1.0.3</code></li>' .
          '</ul>' .
          '<div class="nexaguard-card__footer"><a class="nexaguard-link" href="' . $homeDebug . '" target="_blank" rel="noopener">Run homepage check with debug</a></div>',
      ],
    ];

    // Help section.
    $form['nexaguard_admin']['container']['grid']['help'] = [
      '#type' => 'container',
      '#attributes' => ['class' => ['nexaguard-card']],
      'title' => ['#markup' => '<h2 class="nexaguard-card__title">Need help?</h2>'],
      'desc'  => ['#markup' => '<p class="nexaguard-card__desc">Having trouble with overlay priority, cache, or banner placement? Use the Debug view to verify the loader tag and event flow.</p>'],
      'bullets' => [
        '#markup' =>
          '<ul class="nexaguard-bullets">' .
          '<li>Enable <code>Debug Mode</code> or use the Debug link above.</li>' .
          '<li>Use <em>Reset Consent</em> to clear cookies/storage and hard-reload.</li>' .
          '<li>Purge CDN/page caches if you still see stale markup.</li>' .
          '</ul>',
      ],
    ];

    // Hide default footer actions.
    $form['actions']['#access'] = FALSE;

    return parent::buildForm($form, $form_state);
  }

  /**
   * Validates the configuration form inputs.
   */
  public function validateForm(array &$form, FormStateInterface $form_state): void {
    $id = (string) $form_state->getValue('settings_id');
    if (!preg_match('/^[A-Za-z0-9_\-:.]+$/', $id)) {
      $form_state->setErrorByName('settings_id', $this->t('Settings ID contains invalid characters.'));
    }
  }

  /**
   * Submits and saves configuration form values.
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    $this->configFactory()->getEditable('nexaguard_cmp.settings')
      ->set('settings_id', trim((string) $form_state->getValue('settings_id')))
      ->set('auto_inject', (bool) $form_state->getValue('auto_inject'))
      ->set('enable_gcm', (bool) $form_state->getValue('enable_gcm'))
      ->set('debug_mode', (bool) $form_state->getValue('debug_mode'))
      ->set('anonymous_only', (bool) $form_state->getValue('anonymous_only'))
      ->set('exclude_admin', (bool) $form_state->getValue('exclude_admin'))
      ->save();

    parent::submitForm($form, $form_state);
    $this->messenger()->addStatus($this->t('NexaGuard CMP settings saved.'));
  }

  /**
   * Resets NexaGuard CMP settings to defaults.
   */
  public function resetDefaults(array &$form, FormStateInterface $form_state): void {
    $this->configFactory()->getEditable('nexaguard_cmp.settings')
      ->set('loader_url', 'https://cmp.nexaguard.com/loader.js')
      ->set('api_url', 'https://api.nexaguard.io')
      ->set('assets_url', 'https://cmp.nexaguard.com')
      ->set('auto_inject', TRUE)
      ->set('enable_gcm', TRUE)
      ->set('debug_mode', FALSE)
      ->set('anonymous_only', FALSE)
      ->set('exclude_admin', TRUE)
      ->save();

    $this->messenger()->addStatus($this->t('Defaults restored (except Settings ID).'));
  }

  /**
   * Escapes output for safe HTML rendering.
   */
  private function e(string $v): string {
    return htmlspecialchars($v, ENT_QUOTES, 'UTF-8');
  }

}
