<?php

declare(strict_types=1);

namespace Drupal\nexi_xpay\Form;

use Drupal\Core\Entity\ContentEntityForm;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Url;
use Drupal\nexi_xpay\Plugin\XpayMode\NexiXpayModeManager;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Form controller for the NexiXpayTransaction entity edit forms.
 */
final class NexiXpayTransactionForm extends ContentEntityForm {

  /**
   * The mode manager.
   */
  private NexiXpayModeManager $modeManager;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): self {
    /** @var self $form */
    $form = parent::create($container);
    /** @var NexiXpayModeManager $form['modeManager'] */
    $form->modeManager = $container->get('nexi_xpay.mode_manager');
    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    $form = parent::buildForm($form, $form_state);

    // Hide token fields on creation form (they are auto-generated).
    $is_new = $this->entity->isNew();
    if ($is_new) {
      unset($form['public_token']);
      unset($form['public_token_hash']);
    }
    else {
      // Make token fields read-only on edit form.
      foreach (['public_token', 'public_token_hash'] as $field_name) {
        if (isset($form[$field_name]['widget'][0]['value'])) {
          $form[$field_name]['widget'][0]['value']['#disabled'] = TRUE;
        }
      }

      $token = (string) ($this->entity->get('public_token')->value ?? '');
      if ($token !== '') {
        $pay_url = Url::fromRoute('nexi_xpay.pay', [
          'nexi_xpay_transaction' => $this->entity->id(),
        ], [
          'absolute' => TRUE,
          'query' => ['t' => $token],
        ])->toString();

        $form['pay_link'] = [
          '#type' => 'item',
          '#title' => $this->t('Pay link'),
          '#markup' => '<a href="' . $pay_url . '" target="_blank" rel="noopener noreferrer">' . $pay_url . '</a>',
          '#weight' => 4.5,
        ];
      }
    }

    // Default token expires
    if ($is_new && isset($form['token_expires']['widget'][0]['value'])) {
      // Prefill only if empty.
      $current = $form['token_expires']['widget'][0]['value']['#default_value'] ?? NULL;
      if (empty($current)) {
        // TODO: questi valori non vengono impostati correttamente nella UI di backend
        //      guarda in 'web/core/lib/Drupal/Core/Datetime/Plugin/Field/FieldWidget/TimestampDatetimeWidget.php::formElement()'
        $form['token_expires']['widget'][0]['value']['#default_value'] = \time() + (8 * 60 * 60);
        //$form['token_expires']['widget'][0]['value']['#date_year_range'] = '2025:2027';
      }
      $form['token_expires']['widget'][0]['value']['#description'] = $this->t('If left empty, it will default to 8 hours from now.');
    }

    // Convert field widget to select using available mode plugins.
    if (isset($form['mode']['widget'][0]['value'])) {
      $options = [];
      foreach ($this->modeManager->getDefinitions() as $plugin_id => $definition) {
        $modePluginLabel = (string) ($definition['label'] ?? $plugin_id);
        $modePluginLabel .= isset($definition['description']) ? ' - ' . $definition['description'] : '';
        $options[$plugin_id] = $modePluginLabel;
      }
      asort($options);

      $form['mode']['widget'][0]['value']['#type'] = 'select';
      $form['mode']['widget'][0]['value']['#options'] = $options;
      $form['mode']['widget'][0]['value']['#empty_option'] = $this->t('- Select a payment mode -');
      $form['mode']['widget'][0]['value']['#description'] = $this->t('Available XPay modes provided by enabled sub-modules.');

      // FORCE dropdown rendering (not listbox).
      $form['mode']['widget'][0]['value']['#size'] = 1;
      $form['mode']['widget'][0]['value']['#multiple'] = FALSE;
      unset($form['mode']['widget'][0]['value']['#attributes']['multiple']);
    }

    return $form;
  }

  /**
   * @inheritDoc
   */
  public function validateForm(array &$form, FormStateInterface $form_state): void {
    parent::validateForm($form, $form_state);

    $mode = (string) $form_state->getValue(['mode', 0, 'value'], '');
    if ($mode === '') {
      $form_state->setErrorByName('mode', $this->t('Please select a payment mode.'));
      return;
    }

    if (!$this->modeManager->hasDefinition($mode)) {
      $form_state->setErrorByName('mode', $this->t('Invalid mode selected.'));
    }
  }

  /**
   * {@inheritdoc}
   */
  public function save(array $form, FormStateInterface $form_state): int {
    $is_new = $this->entity->isNew();
    $status = parent::save($form, $form_state);

    if ($is_new) {
      $token = (string) ($this->entity->get('public_token')->value ?? '');
      if ($token !== '') {
        $pay_url = Url::fromRoute('nexi_xpay.pay', [
          'nexi_xpay_transaction' => $this->entity->id(),
        ], [
          'absolute' => TRUE,
          'query' => ['t' => $token],
        ])->toString();

        $this->messenger()->addStatus($this->t('Pay link: @url', ['@url' => $pay_url]));
      }
    }

    $this->messenger()->addStatus($this->t('Transaction saved.'));
    $form_state->setRedirect('entity.nexi_xpay_transaction.collection');
    return $status;
  }

}
