<?php

declare(strict_types=1);

namespace Drupal\nexi_xpay\ListBuilder;

use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Entity\EntityListBuilder;
use Drupal\Core\Link;
use Drupal\Core\Url;
use Drupal\nexi_xpay\Entity\NexiXpayTransaction;
use Drupal\nexi_xpay\Entity\NexiXpayTransactionInterface;

final class NexiXpayTransactionListBuilder extends EntityListBuilder {

  /**
   * {@inheritdoc}
   */
  public function buildHeader(): array {
    $header['id'] = $this->t('ID');
    $header['status'] = $this->t('Status');
    $header['token_expires'] = $this->t('Expire date');
    $header['mode'] = $this->t('Mode');
    $header['amount'] = $this->t('Amount');
    $header['merchant_reference'] = $this->t('Merchant ref');
    $header['changed'] = $this->t('Updated');
    $header['operations'] = $this->t('Operations');
    return $header + parent::buildHeader();
  }

  /**
   * {@inheritdoc}
   */
  public function buildRow(EntityInterface $entity): array {
    /** @var NexiXpayTransaction $entity */
    $row['id'] = Link::createFromRoute(
      (string) $entity->id(),
      'entity.nexi_xpay_transaction.edit_form',
      ['nexi_xpay_transaction' => $entity->id()]
    );
    $row['status'] = $entity->get('status')->value;
    $row['token_expires'] = $entity->get('status')->value == NexiXpayTransactionInterface::STATUS_PENDING
        ? \Drupal::service('date.formatter')->format((int) $entity->get('token_expires')->value, 'short')
        : '-';
    $row['mode'] = $entity->get('mode')->value;
    $row['amount'] = $entity->get('amount')->value . ' ' . $entity->get('currency')->value;
    $row['merchant_reference'] = $entity->get('merchant_reference')->value;
    $row['changed'] = \Drupal::service('date.formatter')->format((int) $entity->getChangedTime(), 'short');
    $row['operations']['data'] = $this->buildOperations($entity);
    return $row + parent::buildRow($entity);
  }

  /**
   * {@inheritdoc}
   */
  public function getDefaultOperations(EntityInterface $entity): array {
    $operations = parent::getDefaultOperations($entity);

    /** @var NexiXpayTransactionInterface $entity */
    $status = (string) $entity->get('status')->value;
    $token = (string) ($entity->get('public_token')->value ?? '');

    // Link "Paga" solo se pending e token presente.
    if ($status === NexiXpayTransactionInterface::STATUS_PENDING && $token !== '') {
      $operations['pay'] = [
        'title' => $this->t('Pay'),
        'url' => Url::fromRoute('nexi_xpay.pay', [
          'nexi_xpay_transaction' => $entity->id(),
        ], [
          'query' => ['t' => $token],
          'absolute' => TRUE,
        ]),
        'weight' => -20,
        'attributes' => [
          'target' => '_blank',
          'rel' => 'noopener noreferrer',
        ],
      ];
    }

    // Link "Rigenera token" solo se pending.
    if ($status === NexiXpayTransactionInterface::STATUS_PENDING) {
      $operations['regenerate_token'] = [
        'title' => $this->t('Regenerate token'),
        'url' => Url::fromRoute('nexi_xpay.regenerate_token', [
          'nexi_xpay_transaction' => $entity->id(),
        ]),
        'weight' => 5,
      ];
    }

    return $operations;
  }


}
