<?php

declare(strict_types=1);

namespace Drupal\nexi_xpay\Plugin\XpayMode;

use Drupal\Core\Entity\EntityStorageException;
use Drupal\Core\Url;
use Drupal\nexi_xpay\Entity\NexiXpayTransactionInterface;
use Drupal\nexi_xpay\Plugin\XpayMode\Attribute\XpayMode;
use Drupal\nexi_xpay\Service\NexiXpayClientInterface;
use Drupal\nexi_xpay\Value\ModeHandleResult;
use Drupal\nexi_xpay\Value\ModeStartResult;
use Symfony\Component\HttpFoundation\RedirectResponse;
use Drupal\Core\Routing\TrustedRedirectResponse;
use Symfony\Component\HttpFoundation\Request;

#[XpayMode(
  id: "hpp_redirect",
  label: "HPP Redirect",
  description: "Hosted Payment Page / browser redirect (server-side)."
)]
final class HostedPaymentPageMode extends NexiXpayModeBase {

  private NexiXpayClientInterface $client;

  public function __construct(array $configuration, $plugin_id, $plugin_definition) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
    if (!isset($configuration['client']) || !$configuration['client'] instanceof NexiXpayClientInterface) {
      throw new \InvalidArgumentException('Missing NexiXpayClientInterface in plugin configuration.');
    }
    $this->client = $configuration['client'];
  }

  public function buildPayPage(NexiXpayTransactionInterface $transaction, Request $request): array {
    $token = (string) $request->query->get('t', '');

    $start_url = Url::fromRoute('nexi_xpay.start', [
      'nexi_xpay_transaction' => $transaction->id(),
    ], [
      'absolute' => TRUE,
      'query' => ['t' => $token],
    ])->toString();

    return [
      '#theme' => 'nexi_xpay_hpp_pay',
      '#tx' => [
        'id' => (int) $transaction->id(),
        'status' => $transaction->getStatus(),
        'amount' => (int) $transaction->getAmount(),
        'currency' => $transaction->getCurrency(),
        'merchant_reference' => $transaction->getMerchantReference(),
      ],
      '#start_url' => $start_url,
      '#cache' => ['max-age' => 0],
    ];
  }

  public function buildReturnPage(NexiXpayTransactionInterface $transaction, Request $request, ModeHandleResult $handleResult): array {
    return [
      '#theme' => 'nexi_xpay_hpp_return',
      '#tx' => [
        'id' => (int) $transaction->id(),
        'status' => $transaction->getStatus(),
      ],
      '#message' => $handleResult->message ?: 'Return handled.',
      '#cache' => ['max-age' => 0],
    ];
  }

  /**
   * @throws EntityStorageException
   */
  public function startPayment(NexiXpayTransactionInterface $transaction, Request $request): ModeStartResult {
    $token = (string) $request->query->get('t', '');
    $orderId = $transaction->getMerchantReference();

    // Browser return/cancel MUST be public and MUST NOT contain token/querystring.
    $returnUrl = Url::fromRoute('nexi_xpay.return', [
      'orderId' => $orderId,
    ], ['absolute' => TRUE])->toString();

    $cancelUrl = Url::fromRoute('nexi_xpay.cancel', [
      'orderId' => $orderId,
    ], ['absolute' => TRUE])->toString();

    // Server-to-server callback.
    $notifyUrl = Url::fromRoute('nexi_xpay.notify', [
      'nexi_xpay_transaction' => $transaction->id(),
    ], ['absolute' => TRUE])->toString();

    // Basic validation for Nexi constraints.
    if ($orderId === '' || strlen($orderId) > 18) {
      $transaction->set('last_error', 'Invalid merchant_reference/orderId (required, max 18 chars).');
      $transaction->save();
      return ModeStartResult::render([
        '#markup' => 'Unable to start payment: invalid orderId.',
        '#cache' => ['max-age' => 0],
      ]);
    }
    if ($transaction->getAmount() <= 0) {
      $transaction->set('last_error', 'Invalid amount (must be > 0).');
      $transaction->save();
      return ModeStartResult::render([
        '#markup' => 'Unable to start payment: invalid amount.',
        '#cache' => ['max-age' => 0],
      ]);
    }
    if ($token === '') {
      // Should not happen if accessed via pay link, but keep it explicit.
      $transaction->set('last_error', 'Missing public token on startPayment().');
      $transaction->save();
      return ModeStartResult::render([
        '#markup' => 'Missing token.',
        '#cache' => ['max-age' => 0],
      ]);
    }

    // Build payload as per Nexi POST /orders/hpp.
    // Minimal required fields:
    // - order: { orderId, amount, currency }
    // - paymentSession: { actionType, amount, language }
    // - resultUrl, cancelUrl
    // - notificationUrl (recommended for your flow)
    $payload = [
      'order' => [
        'orderId' => $orderId,
        'amount' => (string) $transaction->getAmount(),
        'currency' => $transaction->getCurrency(),
        // Optional but useful:
        'description' => 'Nexi XPay transaction ' . (string) $transaction->id(),
      ],
      'paymentSession' => [
        'actionType' => 'PAY',
        'amount' => (string) $transaction->getAmount(),
        // ISO 639-2; default in docs is "ita".
        'language' => 'ita',
        'resultUrl' => $returnUrl,
        'cancelUrl' => $cancelUrl,
        'notificationUrl' => $notifyUrl,
      ],
    ];

    try {
      // Call Nexi (base_url already points to .../api/v1, so path is /orders/hpp).
      $res = $this->client->requestJson('POST', '/orders/hpp', [
        'json' => $payload,
        'timeout' => 20,
        'connect_timeout' => 10,
      ]);

      $status = (int) ($res['status'] ?? 0);
      $data = $res['json'] ?? [];

      if ($status !== 200 || !is_array($data)) {
        $transaction->set('last_error', 'Nexi /orders/hpp unexpected status/body. HTTP ' . $status);
        $transaction->set('raw_response', mb_substr((string) ($res['body'] ?? ''), 0, 60000));
        $transaction->save();

        return ModeStartResult::render([
          '#markup' => 'Unable to start payment (gateway error).',
          '#cache' => ['max-age' => 0],
        ]);
      }

      $hostedPage = (string) ($data['hostedPage'] ?? '');
      $securityToken = (string) ($data['securityToken'] ?? '');

      if ($hostedPage === '' || $securityToken === '') {
        $transaction->set('last_error', 'Nexi /orders/hpp missing hostedPage/securityToken.');
        $transaction->set('raw_response', mb_substr((string) ($res['body'] ?? ''), 0, 60000));
        $transaction->save();

        return ModeStartResult::render([
          '#markup' => 'Unable to start payment (invalid gateway response).',
          '#cache' => ['max-age' => 0],
        ]);
      }

      // Persist correlation fields.
      $transaction->set('order_id', $orderId);          // (o solo se vuoto)
      $transaction->set('security_token', $securityToken);
      if ($transaction->getStatus() === NexiXpayTransactionInterface::STATUS_PENDING) {
        $transaction->setStatus(NexiXpayTransactionInterface::STATUS_PROCESSING);
      }

      // Save logs
      if ($this->client->shouldLogPayloads()) {
        $transaction->set('raw_request', mb_substr(json_encode($payload, JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE), 0, 60000));
        $transaction->set('raw_response', mb_substr((string) ($res['body'] ?? ''), 0, 60000));
      }

      $transaction->save();

      return ModeStartResult::response(new TrustedRedirectResponse($hostedPage));
    }
    catch (\Throwable $e) {
      $transaction->set('last_error', 'Exception during Nexi /orders/hpp: ' . $e->getMessage());
      $transaction->save();

      return ModeStartResult::render([
        '#markup' => 'Unable to start payment (exception).',
        '#cache' => ['max-age' => 0],
      ]);
    }
  }

  /**
   * @throws EntityStorageException
   */
  public function handleReturn(NexiXpayTransactionInterface $transaction, Request $request): ModeHandleResult {
    $current = $transaction->getStatus();

    // 0) Final states: do nothing (idempotent and safe).
    if ($this->isFinalStatus($current)) {
      return ModeHandleResult::noop('Return received (already final).', [
        'source' => 'return',
        'currentStatus' => $current,
        'query' => $request->query->all(),
      ]);
    }

    // 1) Determine orderId for GET /orders/{orderId}.
    // Prefer explicit order_id field; fallback to merchant_reference.
    $orderId = (string) ($transaction->get('order_id')->value ?? '');
    if ($orderId === '') {
      $orderId = $transaction->getMerchantReference();
    }
    $orderId = trim($orderId);

    if ($orderId === '') {
      // No orderId: we can only move pending -> processing.
      if ($current === NexiXpayTransactionInterface::STATUS_PENDING) {
       return ModeHandleResult::status(
         NexiXpayTransactionInterface::STATUS_PROCESSING,
         'Return received. Awaiting confirmation from the payment gateway.',
         [
           'source' => 'return',
           'query' => $request->query->all(),
           'note' => 'Missing orderId; skipping GET /orders/{orderId}.',
         ]
       );
      }
      return ModeHandleResult::noop('Return received (missing orderId).', [
        'source' => 'return',
        'currentStatus' => $current,
        'query' => $request->query->all(),
      ]);
    }

    // 2) Try to retrieve the authoritative outcome via GET /orders/{orderId}.
    try {
      $resp = $this->client->getOrder($orderId,[
        'timeout' => 8,
        'connect_timeout' => 5,
      ]);

      // Optional audit: store raw response (truncate to avoid DB bloat).
      // Salva raw_response solo se abilitato.
      if ($this->client->shouldLogPayloads()) {
        $transaction->set('raw_response', mb_substr((string) ($resp['body'] ?? ''), 0, 60000));
        $transaction->save();
      }

      $data = (array) ($resp['json'] ?? []);
      $outcome = $this->extractOutcomeFromGetOrder($data);
      $mapped = $this->mapOutcomeToInternalStatus($outcome);

      // Se non riesco a mappare, resto in processing (se era pending).
      if ($mapped === NULL) {
        if ($current === NexiXpayTransactionInterface::STATUS_PENDING) {
          return ModeHandleResult::status(
            NexiXpayTransactionInterface::STATUS_PROCESSING,
            'Return received. Awaiting confirmation from the payment gateway...',
            [
              'source' => 'return_get_order',
              'orderId' => $orderId,
              'outcome' => $outcome,
              'query' => $request->query->all(),
            ]
          );
        }
        return ModeHandleResult::noop(
          'Return received. Order not final yet.',
          [
            'source' => 'return_get_order',
            'orderId' => $orderId,
            'outcome' => $outcome,
            'currentStatus' => $current,
            'query' => $request->query->all(),
          ]
        );
      }

      // Applica transition (la state machine e idempotenza la gestisci già col manager/updater).
      return ModeHandleResult::status(
        $mapped,
        'Return processed via GET /orders.',
        [
          'source' => 'return_get_order',
          'orderId' => $orderId,
          'outcome' => $outcome,
          'query' => $request->query->all(),
        ]
      );
    }
    catch (\Throwable $e) {
      // GET failure: do not break UX; move to processing if pending, else noop.
      if (method_exists($transaction, 'set')) {
        $transaction->set('last_error', 'GET /orders failed: ' . $e->getMessage());
        $transaction->save();
      }

      if ($current === NexiXpayTransactionInterface::STATUS_PENDING) {
        return ModeHandleResult::status(
          NexiXpayTransactionInterface::STATUS_PROCESSING,
          'Return received. Awaiting confirmation from the payment gateway...',
          [
            'source' => 'return',
            'orderId' => $orderId,
            'error' => $e->getMessage(),
            'query' => $request->query->all(),
          ]
        );
      }

      return ModeHandleResult::noop('Return received (GET /orders failed).', [
        'source' => 'return',
        'orderId' => $orderId,
        'error' => $e->getMessage(),
        'currentStatus' => $current,
        'query' => $request->query->all(),
      ]);
    }
  }

  private function isFinalStatus(string $status): bool {
    return in_array($status, [
      NexiXpayTransactionInterface::STATUS_PAID,
      NexiXpayTransactionInterface::STATUS_FAILED,
      NexiXpayTransactionInterface::STATUS_CANCELLED,
      NexiXpayTransactionInterface::STATUS_EXPIRED,
    ], TRUE);
  }

  /**
   * Try to extract a meaningful "outcome" from GET /orders/{orderId}.
   * The docs describe an "orderStatus" object but payload details may vary.
   *
   * @return string Uppercase outcome/status candidate, or empty string.
   */
  private function extractOutcomeFromGetOrder(array $data): string {
    $candidates = [];

    // 1) Most important: top-level operations[] (your real payload).
    if (isset($data['operations']) && is_array($data['operations']) && !empty($data['operations'])) {
      $last = end($data['operations']);
      if (is_array($last)) {
        $candidates[] = (string) ($last['operationResult'] ?? '');
        $candidates[] = (string) ($last['operationType'] ?? '');
        $candidates[] = (string) ($last['status'] ?? '');
        if (isset($last['additionalData']) && is_array($last['additionalData'])) {
          $candidates[] = (string) ($last['additionalData']['status'] ?? '');
        }
      }
    }

    // 2) Some variants put operations under orderStatus.
    if (isset($data['orderStatus']) && is_array($data['orderStatus'])) {
      $os = $data['orderStatus'];

      // Sometimes there are explicit status fields (not in your sample, but keep for robustness).
      $candidates[] = (string) ($os['status'] ?? '');
      $candidates[] = (string) ($os['paymentStatus'] ?? '');
      $candidates[] = (string) ($os['result'] ?? '');

      if (isset($os['operations']) && is_array($os['operations']) && !empty($os['operations'])) {
        $last = end($os['operations']);
        if (is_array($last)) {
          $candidates[] = (string) ($last['operationResult'] ?? '');
          if (isset($last['additionalData']) && is_array($last['additionalData'])) {
            $candidates[] = (string) ($last['additionalData']['status'] ?? '');
          }
        }
      }
    }

    // 3) Fallbacks.
    if (isset($data['order']) && is_array($data['order'])) {
      $candidates[] = (string) ($data['order']['status'] ?? '');
      $candidates[] = (string) ($data['order']['result'] ?? '');
    }

    foreach ($candidates as $c) {
      $c = strtoupper(trim($c));
      if ($c !== '') {
        return $c;
      }
    }

    return '';
  }

  /**
   * Map Nexi outcome/order status into internal transaction status.
   */
  private function mapOutcomeToInternalStatus(string $outcome): ?string {
    $o = strtoupper(trim($outcome));
    if ($o === '') {
      return NULL;
    }

    // Final: paid
    if (in_array($o, ['EXECUTED', 'CAPTURED', 'SUCCESS', 'OK'], TRUE)) {
      return NexiXpayTransactionInterface::STATUS_PAID;
    }

    // Non final (in elaborazione)
    if (in_array($o, ['AUTHORIZED', 'PENDING', 'PROCESSING', 'IN_PROGRESS', 'CREATED', 'THREEDS_VALIDATED'], TRUE)) {
      return NexiXpayTransactionInterface::STATUS_PROCESSING;
    }

    // Final: failed
    if (in_array($o, ['DECLINED', 'DENIED_BY_RISK', 'FAILED', 'KO', 'THREEDS_FAILED', 'ERROR'], TRUE)) {
      return NexiXpayTransactionInterface::STATUS_FAILED;
    }

    // Final: cancelled
    if (in_array($o, ['CANCELED', 'CANCELLED', 'VOIDED', 'ABORTED'], TRUE)) {
      return NexiXpayTransactionInterface::STATUS_CANCELLED;
    }

    return NULL;
  }

  /**
   * @throws EntityStorageException
   */
  public function handleNotify(NexiXpayTransactionInterface $transaction, Request $request): ModeHandleResult {
    // 1) Parse payload: prefer JSON, fallback to form params.
    $raw = (string) $request->getContent();
    $data = [];
    $content_type = (string) $request->headers->get('content-type', '');

    if (str_contains($content_type, 'application/json')) {
      $decoded = json_decode($raw, TRUE);
      if (is_array($decoded)) {
        $data = $decoded;
      }
    }
    if (!$data) {
      // Fallback: x-www-form-urlencoded or multipart.
      $data = $request->request->all();
    }

    // Persist raw payload for audit/debug (truncate to avoid huge rows).
    if ($this->client->shouldLogPayloads()) {
      if ($raw === '' && $data) {
        $raw = json_encode($data, JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE);
      }
      if ($raw !== '') {
        $transaction->set('raw_request', mb_substr($raw, 0, 60000));
      }
    }

    // 2) Extract standard fields.
    $securityToken = (string) ($data['securityToken'] ?? '');
    $eventId = (string) ($data['eventId'] ?? '');
    $operation = is_array($data['operation'] ?? NULL) ? $data['operation'] : [];
    $orderId = (string) ($operation['orderId'] ?? '');
    $operationResult = strtoupper((string) ($operation['operationResult'] ?? ''));

    // 3) Security token validation (if configured).
    $expected = (string) ($transaction->get('security_token')->value ?? '');
    if ($expected !== '' && $securityToken !== '' && !hash_equals($expected, $securityToken)) {
      $transaction->set('last_error', 'Notify securityToken mismatch.');
      $transaction->save();

      return ModeHandleResult::noop('Notify ignored: security token mismatch.', [
        'eventId' => $eventId,
        'orderId' => $orderId,
        'operationResult' => $operationResult,
      ]);
    }

    // Strict mode: security_token must be set during start/init (Step 3).
    if ($expected === '') {
      $transaction->set('last_error', 'Notify received but security_token not configured.');
      $transaction->save();

      return ModeHandleResult::noop('Notify ignored: security token not configured.', [
        'eventId' => $eventId,
        'orderId' => $orderId,
        'operationResult' => $operationResult,
      ]);
    }

    // 4) Store orderId if present and missing.
    $currentOrderId = (string) ($transaction->get('order_id')->value ?? '');
    if ($currentOrderId === '' && $orderId !== '') {
      $transaction->set('order_id', $orderId);
    }

    // 5) Map operationResult -> internal status.
    $newStatus = $this->mapOperationResultToStatus($operationResult);

    // If we can't map, just ack idempotently.
    if ($newStatus === NULL) {
      $transaction->save();
      return ModeHandleResult::noop('Notify received (unmapped).', [
        'eventId' => $eventId,
        'orderId' => $orderId,
        'operationResult' => $operationResult,
      ]);
    }

    // 6) Idempotent + monotonic transitions.
    $current = $transaction->getStatus();
    $transaction->save();
    if (!$this->isTransitionAllowed($current, $newStatus)) {
      return ModeHandleResult::noop('Notify received (ignored by state machine).', [
        'eventId' => $eventId,
        'orderId' => $orderId,
        'operationResult' => $operationResult,
        'currentStatus' => $current,
        'newStatus' => $newStatus,
      ]);
    }

    return ModeHandleResult::status($newStatus, 'Notify processed.', [
      'eventId' => $eventId,
      'orderId' => $orderId,
      'operationResult' => $operationResult,
    ]);
  }

  /**
   * Conservative mapping from Nexi operationResult to internal transaction status.
   */
  private function mapOperationResultToStatus(string $operationResult): ?string {
    return match ($operationResult) {
      'EXECUTED' => NexiXpayTransactionInterface::STATUS_PAID,

      // Authorization/3DS steps: keep it non-final.
      'AUTHORIZED', 'PENDING', 'THREEDS_VALIDATED' => NexiXpayTransactionInterface::STATUS_PROCESSING,

      // Negative outcomes.
      'DECLINED', 'DENIED_BY_RISK', 'THREEDS_FAILED', 'FAILED' => NexiXpayTransactionInterface::STATUS_FAILED,

      // Customer/system cancellation-like outcomes.
      'CANCELED', 'VOIDED' => NexiXpayTransactionInterface::STATUS_CANCELLED,

      default => NULL,
    };
  }

  /**
   * Prevent non-monotonic transitions (idempotent and safe).
   */
  private function isTransitionAllowed(string $current, string $next): bool {
    // The same state is always idempotently OK (updater will no-op anyway).
    if ($current === $next) {
      return TRUE;
    }

    // Final states: do not change them via notify.
    $final = [
      NexiXpayTransactionInterface::STATUS_PAID,
      NexiXpayTransactionInterface::STATUS_FAILED,
      NexiXpayTransactionInterface::STATUS_CANCELLED,
      NexiXpayTransactionInterface::STATUS_EXPIRED,
    ];
    if (in_array($current, $final, TRUE)) {
      return FALSE;
    }

    // From pending/processing you can go to anything mapped.
    return in_array($current, [
      NexiXpayTransactionInterface::STATUS_PENDING,
      NexiXpayTransactionInterface::STATUS_PROCESSING,
    ], TRUE);
  }

}
