<?php

declare(strict_types=1);

namespace Drupal\nexi_xpay\Service;

use Drupal\Core\Entity\EntityStorageException;
use Drupal\nexi_xpay\Entity\NexiXpayTransactionInterface;
use Drupal\nexi_xpay\Event\NexiXpayEvents;
use Drupal\nexi_xpay\Event\TransactionStatusChangeEvent;
use Psr\Log\LoggerInterface;
use Symfony\Contracts\EventDispatcher\EventDispatcherInterface;

final readonly class TransactionStatusUpdater {

  public function __construct(
    private EventDispatcherInterface $dispatcher,
    private LoggerInterface          $logger,
  ) {}

  /**
   * @throws EntityStorageException
   */
  public function setStatusIfChanged(NexiXpayTransactionInterface $transaction, string $newStatus, array $context = []): bool {
    $old = $transaction->getStatus();
    if ($old === $newStatus) {
      return FALSE;
    }
    $transaction->setStatus($newStatus);
    $transaction->save();

    $this->dispatcher->dispatch(
      new TransactionStatusChangeEvent($transaction, $old, $newStatus, $context),
      NexiXpayEvents::TRANSACTION_STATUS_CHANGE
    );

    $this->logger->info('Transaction {id} status change {old} -> {new}', [
        'id' => $transaction->id(),
        'old' => $old,
        'new' => $newStatus,
      ] + $context);

    return TRUE;
  }

}
