<?php

declare(strict_types=1);

namespace Drupal\nexi_xpay\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;

/**
 * The class NexiXpaySettingsForm manages the configuration form.
 */
final class NexiXpaySettingsForm extends ConfigFormBase {

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'nexi_xpay_settings_form';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames(): array {
    return ['nexi_xpay.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    $config = $this->config('nexi_xpay.settings');

    $form['#tree'] = TRUE;

    $form['environment'] = [
      '#type' => 'select',
      '#title' => $this->t('Environment'),
      '#description' => $this->t('Select the environment where the plugin should be used. For details check <a href="https://developer.nexi.it/en/api/specifiche-di-utilizzo-del-servizio" title="Service usage specifications" target="_blank">this page</a>.'),
      '#options' => [
        'test' => $this->t('Test'),
        'production' => $this->t('Production'),
      ],
      '#default_value' => $config->get('environment') ?? 'test',
      '#required' => TRUE,
    ];

    $form['test'] = [
      '#type' => 'details',
      '#title' => $this->t('Test settings'),
      '#description' => $this->t('These settings are used when the selected environment is set to <em>Test</em>. For details check <a href="https://developer.nexi.it/en/area-test/api-key" title="Test API Keys" target="_blank">this page</a>.'),
      '#open' => TRUE,
    ];
    $form['test']['base_url'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Test gateway URL'),
      '#default_value' => $config->get('test.base_url') ?? '',
      '#required' => FALSE,
    ];
    $form['test']['api_key'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Test API KEY'),
      '#default_value' => $config->get('test.api_key') ?? '',
      '#required' => FALSE,
    ];
    $form['test']['merchant_id'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Test merchant id (or terminal)'),
      '#default_value' => $config->get('test.merchant_id') ?? '',
      '#required' => FALSE,
    ];

    $form['production'] = [
      '#type' => 'details',
      '#title' => $this->t('Production settings'),
      '#open' => TRUE,
    ];
    $form['production']['base_url'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Production gateway URL'),
      '#default_value' => $config->get('production.base_url') ?? '',
      '#required' => FALSE,
    ];
    $form['production']['api_key'] = [
      '#type' => 'password',
      '#title' => $this->t('Production API KEY'),
      '#default_value' => '',
      '#required' => FALSE,
      '#description' => $this->t('Leave empty to keep the currently saved value.'),
    ];
    $form['production']['merchant_id'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Production merchant id (or terminal)'),
      '#default_value' => $config->get('production.merchant_id') ?? '',
      '#required' => FALSE,
    ];

    $form['logging'] = [
      '#type' => 'details',
      '#title' => $this->t('Logging settings'),
      '#open' => FALSE,
    ];

    $form['logging']['log_payloads'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Log payloads'),
      '#description' => $this->t('Log HTTP requests and responses for debugging purposes.'),
      '#default_value' => $config->get('logging.log_payloads') ?? FALSE,
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state): void {
    parent::validateForm($form, $form_state);

    // Validate mandatory fields.
    $environment = $form_state->getValue('environment');

    if ($environment === 'test') {
      if (empty($form_state->getValue(['test', 'base_url']))) {
        $form_state->setErrorByName('test][base_url', $this->t('Test gateway URL is required.'));
      }
      if (empty($form_state->getValue(['test', 'api_key']))) {
        $form_state->setErrorByName('test][api_key', $this->t('Test API KEY is required.'));
      }
      if (empty($form_state->getValue(['test', 'merchant_id']))) {
        $form_state->setErrorByName('test][merchant_id', $this->t('Test merchant id is required.'));
      }
    }
    else {
      if (empty($form_state->getValue(['production', 'base_url']))) {
        $form_state->setErrorByName('production][base_url', $this->t('Production gateway URL is required.'));
      }
      if (empty($form_state->getValue(['production', 'api_key']))) {
        $form_state->setErrorByName('production][api_key', $this->t('Production API KEY is required.'));
      }
      if (empty($form_state->getValue(['production', 'merchant_id']))) {
        $form_state->setErrorByName('production][merchant_id', $this->t('Production merchant id is required.'));
      }
    }
  }

  /**
   * @inheritDoc
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {

    $this->configFactory->getEditable('nexi_xpay.settings')
      ->set('environment', $form_state->getValue('environment'))
      ->set('test.base_url', $form_state->getValue(['test', 'base_url']))
      ->set('test.api_key', $form_state->getValue(['test', 'api_key']))
      ->set('test.merchant_id', $form_state->getValue(['test', 'merchant_id']))
      ->set('production.base_url', $form_state->getValue(['production', 'base_url']))
      ->set('production.merchant_id', $form_state->getValue(['production', 'merchant_id']))
      ->set('logging.log_payloads', $form_state->getValue(['logging', 'log_payloads']))
      ->save();

    // Update secrets only if a new value was provided.
    $prod_api_key = trim((string) $form_state->getValue(['production', 'api_key']));
    if ($prod_api_key !== '') {
      $this->configFactory->getEditable('nexi_xpay.settings')
        ->set('production.api_key', $prod_api_key)
        ->save();
    }

    parent::submitForm($form, $form_state);
    $this->messenger()->addStatus($this->t('Settings saved.'));
  }
}
