<?php

declare(strict_types=1);

namespace Drupal\nexi_xpay\Access;

use Drupal\Core\Access\AccessResult;
use Drupal\Core\Session\AccountInterface;
use Drupal\nexi_xpay\Entity\NexiXpayTransactionInterface;
use Symfony\Component\HttpFoundation\Request;

/**
 * Access check for Nexi notify callbacks using a dedicated per-transaction token.
 *
 * This check is intentionally simple and cache-disabled.
 * It validates a plain token coming from the URL path against the stored hash.
 */
final class NotifyTokenAccessCheck {

  /**
   * Custom access callback for the notify route.
   *
   * Route signature:
   *   /nexi-xpay/notify/{nexi_xpay_transaction}/{token}
   */
  public static function access(
    AccountInterface $account,
    NexiXpayTransactionInterface $nexi_xpay_transaction,
    string $token,
    Request $request,
  ): AccessResult {
    // Token is required and must be non-empty.
    $token = trim($token);
    if ($token === '') {
      return AccessResult::forbidden('Missing token.')->setCacheMaxAge(0);
    }

    $hash = $nexi_xpay_transaction->getNotifyTokenHash();
    if ($hash === NULL || $hash === '') {
      // If no hash is stored, deny access (misconfigured transaction).
      return AccessResult::forbidden('Notify token not configured.')->setCacheMaxAge(0);
    }

    // Optional: reuse the existing expiry window if configured on the transaction.
    // This keeps behavior consistent with the public token access check.
    $expires = $nexi_xpay_transaction->getTokenExpiresTime();
    if ($expires !== NULL && \time() > $expires) {
      return AccessResult::forbidden('Token expired.')->setCacheMaxAge(0);
    }

    $calc = hash('sha256', $token);
    if (!hash_equals($hash, $calc)) {
      return AccessResult::forbidden('Invalid token.')->setCacheMaxAge(0);
    }

    return AccessResult::allowed()->setCacheMaxAge(0);
  }

}
