<?php

declare(strict_types=1);

namespace Drupal\nexi_xpay\Controller;

use Drupal\Component\Plugin\Exception\InvalidPluginDefinitionException;
use Drupal\Component\Plugin\Exception\PluginNotFoundException;
use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Entity\EntityStorageException;
use Drupal\nexi_xpay\Entity\NexiXpayTransactionInterface;
use Drupal\nexi_xpay\Service\NexiXpayManagerInterface;
use Drupal\nexi_xpay\Service\TransactionStatusUpdater;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\HttpKernel\Exception\NotFoundHttpException;

final class PaymentController extends ControllerBase {

  public function __construct(
    private readonly NexiXpayManagerInterface $manager,
    private readonly TransactionStatusUpdater $statusUpdater,
  ) {}

  public static function create(ContainerInterface $container): self {
    return new self(
      $container->get('nexi_xpay.manager'),
      $container->get('nexi_xpay.status_updater'),
    );
  }

  public function start(NexiXpayTransactionInterface $nexi_xpay_transaction, Request $request): array|Response|null {
    $result = $this->manager->startPayment($nexi_xpay_transaction, $request);
    if ($result->response) {
      return $result->response;
    }
    // Fallback: if a mode returns render by mistake, render it.
    return $result->renderArray ?? ['#markup' => 'Unable to start payment.'];
  }

  public function pay(NexiXpayTransactionInterface $nexi_xpay_transaction, Request $request): array {
    return $this->manager->buildPayPage($nexi_xpay_transaction, $request);
  }

  /**
   * @throws EntityStorageException
   * @throws InvalidPluginDefinitionException
   * @throws PluginNotFoundException
   */
  public function handleReturnByOrder(string $orderId, Request $request): array {
    $tx = $this->entityTypeManager()
      ->getStorage('nexi_xpay_transaction')
      ->loadByProperties(['order_id' => $orderId]);

    $tx = $tx ? reset($tx) : NULL;
    if (!$tx) {
      throw new NotFoundHttpException();
    }

    // Reuse existing return handler logic (handle + render via plugin).
    return $this->handleReturn($tx, $request);
  }

  /**
   * @throws EntityStorageException
   * @throws InvalidPluginDefinitionException
   * @throws PluginNotFoundException
   */
  public function handleCancelByOrder(string $orderId, Request $request): array {
    $tx = $this->entityTypeManager()
      ->getStorage('nexi_xpay_transaction')
      ->loadByProperties(['order_id' => $orderId]);

    $tx = $tx ? reset($tx) : NULL;
    if (!$tx) {
      throw new NotFoundHttpException();
    }

    // Treat cancel as return; mode can show a "cancelled" UX.
    // For now we just go through return handler; later we can add a dedicated hook.
    return $this->handleReturn($tx, $request);
  }

  /**
   * @throws EntityStorageException
   */
  public function handleReturn(NexiXpayTransactionInterface $nexi_xpay_transaction, Request $request): array {
    $handle = $this->manager->handleReturn($nexi_xpay_transaction, $request);

    if ($handle->newStatus) {
      $this->statusUpdater->setStatusIfChanged($nexi_xpay_transaction, $handle->newStatus, $handle->context);
    }

    return $this->manager->buildReturnPage($nexi_xpay_transaction, $request, $handle);
  }

  /**
   * @throws EntityStorageException
   */
  public function handleNotify(NexiXpayTransactionInterface $nexi_xpay_transaction, Request $request): Response {
    $res = $this->manager->handleNotify($nexi_xpay_transaction, $request);

    if ($res->newStatus) {
      $this->statusUpdater->setStatusIfChanged($nexi_xpay_transaction, $res->newStatus, $res->context);
    }

    // Idempotent webhook endpoint:
    // - Do not leak entity id/status/message.
    // - Keep response constant to avoid information disclosure.
    return new Response('', 204);
  }

}
