<?php

declare(strict_types=1);

namespace Drupal\nexi_xpay\Entity;

use Drupal\Core\Entity\Attribute\ContentEntityType;
use Drupal\Core\Entity\ContentEntityBase;
use Drupal\Core\Entity\EntityTypeInterface;
use Drupal\Core\Field\BaseFieldDefinition;
use Drupal\Core\Entity\EntityStorageInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\nexi_xpay\Util\SecretGenerator;

/**
 * Defines the Nexi Xpay Transaction entity.
 *
 * This content entity is used to manage transactions processed via the Nexi Xpay payment gateway.
 *
 * Key features:
 * - Auto-generation of merchant references.
 * - Auto-generation of public tokens and their corresponding hashes.
 * - Token expiry management.
 * - Tracks transaction states, amounts, and payment modes.
 *
 * Entity metadata:
 * - ID: 'nexi_xpay_transaction'
 * - Label: Translatable 'Nexi Xpay Transaction'
 * - Base table: 'nexi_xpay_transaction'
 * - Entity keys: 'id', 'uuid'
 * - Custom handlers for list builders, forms, and access control.
 * - Admin permission: 'administer nexi-xpay transactions'
 *
 * Links:
 * - Collection: '/admin/content/nexi-xpay/transactions'
 * - Add form: '/admin/content/nexi-xpay/transactions/add'
 * - Edit form: '/admin/content/nexi-xpay/transactions/{nexi_xpay_transaction}/edit'
 * - Delete form: '/admin/content/nexi-xpay/transactions/{nexi_xpay_transaction}/delete'
 *
 * The entity contains the following key properties:
 * - status: Represents the transaction state (e.g., Pending, Paid).
 * - merchant_reference: A unique reference for the transaction set by the merchant.
 * - amount: Amount of the transaction in minor units.
 * - currency: ISO 4217 three-character currency code.
 * - mode: Payment mode identifier (plugin).
 * - public_token and public_token_hash: Token for generating public links and its hashed value.
 * - token_expires: Expiry timestamp of the public token.
 * - order_id: Correlation identifier with the order system.
 *
 * Implements methods for:
 * - Managing transaction states.
 * - Auto-generating merchant references and public tokens.
 * - Configuring base field definitions for the entity properties.
 */
#[ContentEntityType(
  id: 'nexi_xpay_transaction',
  label: new TranslatableMarkup('Nexi Xpay Transaction'),
  entity_keys: [
    'id' => 'id',
    'uuid' => 'uuid',
  ],
  handlers: [
    'list_builder' => 'Drupal\nexi_xpay\ListBuilder\NexiXpayTransactionListBuilder',
    'form' => [
      'add' => 'Drupal\nexi_xpay\Form\NexiXpayTransactionForm',
      'edit' => 'Drupal\nexi_xpay\Form\NexiXpayTransactionForm',
      'delete' => 'Drupal\Core\Entity\ContentEntityDeleteForm',
    ],
    'access' => 'Drupal\Core\Entity\EntityAccessControlHandler',
  ],
  links: [
    'collection' => '/admin/content/nexi-xpay/transactions',
    'add-form' => '/admin/content/nexi-xpay/transactions/add',
    'edit-form' => '/admin/content/nexi-xpay/transactions/{nexi_xpay_transaction}/edit',
    'delete-form' => '/admin/content/nexi-xpay/transactions/{nexi_xpay_transaction}/delete',
  ],
  admin_permission: 'administer nexi-xpay transactions',
  base_table: 'nexi_xpay_transaction'
)]
final class NexiXpayTransaction extends ContentEntityBase implements NexiXpayTransactionInterface {

  use StringTranslationTrait;

  /**
   * @inheritDoc
   */
  public function preSave(EntityStorageInterface $storage): void {
    parent::preSave($storage);

    // Default status: pending if empty.
    if ($this->get('status')->isEmpty()) {
      $this->set('status', self::STATUS_PENDING);
    }

    if ($this->isNew()) {
      // Generate a Nexi-compliant id.
      $generated_order_id = SecretGenerator::generateBase36Id(18);

      if ($this->get('order_id')->isEmpty() || (string) $this->get('order_id')->value === '') {
        $this->set('order_id', $generated_order_id);
      }

      // Auto-generate merchant_reference if empty: max 18, [a-z0-9].
      // In this case merchant_reference will be equal to order_id.
      if ($this->get('merchant_reference')->isEmpty() || $this->get('merchant_reference')->value === '') {
        $this->set('merchant_reference', $generated_order_id);
      }

      // Auto-generate public token and hash if missing.
      $plain = (string) ($this->get('public_token')->value ?? '');
      $hash = (string) ($this->get('public_token_hash')->value ?? '');

      if ($plain === '') {
        $generated_public_token = SecretGenerator::generateHexSecretWithHash(16);
        $this->set('public_token', $generated_public_token['secret']);
        $this->set('public_token_hash', $generated_public_token['hash']);
      }
      elseif ($hash === '') {
        $this->set('public_token_hash', hash('sha256', $plain));
      }

      // Default token expiry: now + 8 hours (creation only) if not provided.
      $expires = $this->get('token_expires')->value;

      // Treat empty / 0 as not set.
      if (empty($expires) || (int) $expires <= 0) {
        $this->set('token_expires', \time() + (8 * 60 * 60));
      }

      // Auto-generate notify token only on first creation if missing.
      // Store token in plain (admin/debug use) and hash for notify access check.
      $notify_plain = (string) ($this->get('notify_token')->value ?? '');
      $notify_hash = (string) ($this->get('notify_token_hash')->value ?? '');

      if ($notify_plain === '') {
        $generated_notify_token = SecretGenerator::generateHexSecretWithHash(32);
        $this->set('notify_token', $generated_notify_token['secret']);
        $this->set('notify_token_hash', $generated_notify_token['hash']);
      }
      elseif ($notify_hash === '') {
        $this->set('notify_token_hash', hash('sha256', $notify_plain));
      }
    }

  }


  /**
   * @inheritDoc
   */
  public static function getStates(): array {
    return [
      self::STATUS_PENDING => new TranslatableMarkup('Pending'),
      self::STATUS_PROCESSING => new TranslatableMarkup('Processing'),
      self::STATUS_PAID => new TranslatableMarkup('Paid'),
      self::STATUS_FAILED => new TranslatableMarkup('Failed'),
      self::STATUS_CANCELLED => new TranslatableMarkup('Cancelled'),
      self::STATUS_EXPIRED => new TranslatableMarkup('Expired'),
    ];
  }

  /**
   * @inheritDoc
   */
  public static function getDefaultStatus(): string {
    return self::STATUS_PENDING;
  }

  /**
   * @inheritDoc
   */
  public function getStatus(): string {
    return (string) $this->get('status')->value;
  }

  /**
  * @inheritDoc
  */
  public function setStatus(string $status): static {
    $this->set('status', $status);
    return $this;
  }

  /**
   * @inheritDoc
   */
  public function getMode(): string {
    return (string) $this->get('mode')->value;
  }

  /**
  * @inheritDoc
  */
  public function setMode(string $mode): static {
    $this->set('mode', $mode);
    return $this;
  }

  /**
   * @inheritDoc
   */
  public function getPublicTokenHash(): ?string {
    $v = $this->get('public_token_hash')->value;
    return $v === NULL ? NULL : (string) $v;
  }

  /**
  * @inheritDoc
  */
  public function setPublicTokenHash(string $hash): static {
    $this->set('public_token_hash', $hash);
    return $this;
  }

  /**
  * @inheritDoc
  */
  public function getTokenExpiresTime(): ?int {
    $v = $this->get('token_expires')->value;
    return $v === NULL ? NULL : (int) $v;
  }

  /**
  * @inheritDoc
  */
  public function setTokenExpiresTime(?int $timestamp): static {
    $this->set('token_expires', $timestamp);
    return $this;
  }

  /**
   * @inheritDoc
   */
  public function getNotifyTokenHash(): ?string {
    $v = $this->get('notify_token_hash')->value;
    return $v === NULL ? NULL : (string) $v;
  }

  /**
   * @inheritDoc
   */
  public function setNotifyTokenHash(string $hash): static {
    $this->set('notify_token_hash', $hash);
    return $this;
  }

  /**
   * @inheritDoc
   */
  public function getNotifyToken(): string {
    return (string) ($this->get('notify_token')->value ?? '');
  }

  /**
   * @inheritDoc
   */
  public function setNotifyToken(string $token): static {
    $this->set('notify_token', $token);
    return $this;
  }


  /**
   * @inheritDoc
   */
  public function getMerchantReference(): string {
    return (string) $this->get('merchant_reference')->value;
  }

  /**
   * @inheritDoc
   */
  public function getAmount(): int {
    return (int) $this->get('amount')->value;
  }

  /**
   * @inheritDoc
   */
  public function getCurrency(): string {
    return (string) $this->get('currency')->value;
  }

  /**
   * @inheritDoc
   */
  public function getOrderId(): string {
    return (string) $this->get('order_id')->value;
  }

  /**
   * {@inheritdoc}
   */
  public static function baseFieldDefinitions(EntityTypeInterface $entity_type): array {
    $fields = parent::baseFieldDefinitions($entity_type);

    $fields['uuid']->setLabel(new TranslatableMarkup('UUID'));

    $fields['status'] = BaseFieldDefinition::create('list_string')
      ->setLabel(new TranslatableMarkup('Status'))
      ->setRequired(TRUE)
      ->setSettings(['allowed_values' => self::getStates()])
      ->setDefaultValue(self::getDefaultStatus())
      ->setDisplayOptions('form', ['type' => 'options_select', 'weight' => 0])
      ->setDisplayConfigurable('form', TRUE);

    $fields['merchant_reference'] = BaseFieldDefinition::create('string')
      ->setLabel(new TranslatableMarkup('Merchant reference'))
      ->setDescription(new TranslatableMarkup('Reference with the merchant system business logic. Auto-generated if empty.'))
      ->setSetting('max_length', 255)
      ->setDisplayOptions('form', ['type' => 'string_textfield', 'weight' => 1])
      ->setDisplayConfigurable('form', TRUE);

    $fields['amount'] = BaseFieldDefinition::create('integer')
      ->setLabel(new TranslatableMarkup('Amount (minor units)'))
      ->setRequired(TRUE)
      ->setDisplayOptions('form', ['type' => 'number', 'weight' => 2])
      ->setDisplayConfigurable('form', TRUE);

    $fields['currency'] = BaseFieldDefinition::create('string')
      ->setLabel(new TranslatableMarkup('Currency'))
      ->setRequired(TRUE)
      ->setSetting('max_length', 3)
      ->setDefaultValue('EUR')
      ->setDisplayOptions('form', ['type' => 'string_textfield', 'weight' => 3])
      ->setDisplayConfigurable('form', TRUE);

    $fields['mode'] = BaseFieldDefinition::create('string')
      ->setLabel(new TranslatableMarkup('Mode'))
      ->setDescription(new TranslatableMarkup('Payment mode plugin id (selected from enabled mode plugins).'))
      ->setRequired(TRUE)
      ->setSetting('max_length', 64)
      ->setDisplayOptions('form', [
        'type' => 'options_select',
        'weight' => 4,
      ])
      ->setDisplayConfigurable('form', TRUE);

    $fields['public_token'] = BaseFieldDefinition::create('string')
      ->setLabel(new TranslatableMarkup('Public token (plain)'))
      ->setDescription(new TranslatableMarkup('Token in chiaro usato per generare il link pubblico. Conservato per uso admin.'))
      ->setSetting('max_length', 64);

    $fields['public_token_hash'] = BaseFieldDefinition::create('string')
      ->setLabel(new TranslatableMarkup('Public token hash'))
      ->setSetting('max_length', 128)
      ->setDescription(new TranslatableMarkup('Store only the hash (SHA-256 hex).'));

    $fields['token_expires'] = BaseFieldDefinition::create('timestamp')
      ->setLabel(new TranslatableMarkup('Token expires'))
      ->setDescription(new TranslatableMarkup('Optional public token expiry timestamp.'))
      ->setDisplayOptions('form', ['type' => 'datetime_timestamp', 'weight' => 6])
      ->setDisplayConfigurable('form', TRUE);

    $fields['notify_token'] = BaseFieldDefinition::create('string')
      ->setLabel(new TranslatableMarkup('Notify token (plain)'))
      ->setDescription(new TranslatableMarkup('Token used in notificationUrl (admin/debug use).'))
      ->setSetting('max_length', 64);

    $fields['notify_token_hash'] = BaseFieldDefinition::create('string')
      ->setLabel(new TranslatableMarkup('Notify token hash'))
      ->setSetting('max_length', 128)
      ->setDescription(new TranslatableMarkup('Store only the hash (SHA-256 hex) for notify access check.'));

    // Nexi correlation fields.
    $fields['order_id'] = BaseFieldDefinition::create('string')
      ->setLabel(new TranslatableMarkup('NEXI Order ID Reference'))
      ->setDescription(new TranslatableMarkup('Max 18, lowercase [a-z0-9]. Auto-generated from merchant_reference if empty.'))
      ->setSetting('max_length', 18)
      ->setDisplayOptions('form', ['type' => 'string_textfield', 'weight' => 7])
      ->setDisplayConfigurable('form', TRUE);

    $fields['session_id'] = BaseFieldDefinition::create('string')
      ->setLabel(new TranslatableMarkup('Session ID'))
      ->setSetting('max_length', 128);

    $fields['security_token'] = BaseFieldDefinition::create('string')
      ->setLabel(new TranslatableMarkup('Security token'))
      ->setSetting('max_length', 128)
      ->setDescription(new TranslatableMarkup('Used to validate notify callbacks (mode-specific).'));

    // Audit/debug.
    $fields['raw_request'] = BaseFieldDefinition::create('string_long')
      ->setLabel(new TranslatableMarkup('Raw request (debug)'))
      ->setDisplayOptions('form', ['type' => 'text_textarea', 'weight' => 10])
      ->setDisplayConfigurable('form', TRUE);

    $fields['raw_response'] = BaseFieldDefinition::create('string_long')
      ->setLabel(new TranslatableMarkup('Raw response (debug)'))
      ->setDisplayOptions('form', ['type' => 'text_textarea', 'weight' => 11])
      ->setDisplayConfigurable('form', TRUE);

    $fields['last_error'] = BaseFieldDefinition::create('string_long')
      ->setLabel(new TranslatableMarkup('Last error'))
      ->setDisplayOptions('form', ['type' => 'text_textarea', 'weight' => 12])
      ->setDisplayConfigurable('form', TRUE);

    $fields['created'] = BaseFieldDefinition::create('created')->setLabel(new TranslatableMarkup('Created'));
    $fields['changed'] = BaseFieldDefinition::create('changed')->setLabel(new TranslatableMarkup('Changed'));

    return $fields;
  }

  /**
   * {@inheritdoc}
   */
  public function getChangedTime(): int {
    return (int) $this->get('changed')->value;
  }

  /**
   * {@inheritdoc}
   */
  public function setChangedTime($timestamp): static {
    $this->set('changed', (int) $timestamp);
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getChangedTimeAcrossTranslations(): int {
    // For this MVP we do not manage translations; return the base changed time.
    // Drupal core expects an int.
    return $this->getChangedTime();
  }

}
