<?php

declare(strict_types=1);

namespace Drupal\nexi_xpay\Entity;

use Drupal\Core\Entity\ContentEntityInterface;
use Drupal\Core\Entity\EntityChangedInterface;

/**
 * Defines the interface for NexiXpayTransaction entities.
 */
interface NexiXpayTransactionInterface extends ContentEntityInterface, EntityChangedInterface {

  /**
   * Status constants.
   */
  public const string STATUS_PENDING = 'pending';
  public const string STATUS_PROCESSING = 'processing';
  public const string STATUS_PAID = 'paid';
  public const string STATUS_FAILED = 'failed';
  public const string STATUS_CANCELLED = 'cancelled';
  public const string STATUS_EXPIRED = 'expired';

  /**
   * Returns the NexiXpayTransaction states.
   *
   * @return array
   */
  public static function getStates(): array;

  /**
   * Returns the default transaction status.
   *
   * @return string
   */
  public static function getDefaultStatus(): string;

  /**
   * Returns the transaction status.
   *
   * @return string
   */
  public function getStatus(): string;

  /**
   * Sets the NexiXpayTransaction status.
   *
   * @param string $status
   *
   * @return static
   */
  public function setStatus(string $status): static;

  /**
   * Returns the payment mode identifier.
   *
   * @return string
   */
  public function getMode(): string;

  /**
   * Sets the payment mode identifier.
   *
   * @param string $mode
   *
   * @return static
   */
  public function setMode(string $mode): static;

  /**
   * Returns the public token.
   *
   * @return string|null
   */
  public function getPublicTokenHash(): ?string;

  /**
   * Sets the public token hash.
   *
   * @param string $hash
   *
   * @return static
   */
  public function setPublicTokenHash(string $hash): static;

  /**
   * Returns the public token expiry timestamp.
   *
   * @return int|null
   */
  public function getTokenExpiresTime(): ?int;

  /**
   * Sets the public token expiry timestamp.
   *
   * @param int|null $timestamp
   *
   * @return static
   */
  public function setTokenExpiresTime(?int $timestamp): static;

  /**
   * Returns the notify token hash (SHA-256 hex).
   */
  public function getNotifyTokenHash(): ?string;

  /**
   * Sets the notify token hash (SHA-256 hex).
   */
  public function setNotifyTokenHash(string $hash): static;

  /**
   * Returns the notify token in plain form (admin/debug use).
   *
   * This value is used to build the notificationUrl sent to Nexi.
   */
  public function getNotifyToken(): string;

  /**
   * Sets the notify token in plain form (admin/debug use).
   */
  public function setNotifyToken(string $token): static;

  /**
   * Returns the merchant reference.
   *
   * @return string
   */
  public function getMerchantReference(): string;

  /**
   * Returns the transaction amount in minor units.
   *
   * @return int
   */
  public function getAmount(): int;

  /**
   * Returns the transaction currency.
   *
   * @return string
   */
  public function getCurrency(): string;

  /**
   * Returns the NEXI Order ID.
   *
   * @return string
   */
  public function getOrderId(): string;

}
