<?php

declare(strict_types=1);

namespace Drupal\nexi_xpay\Form;

use Drupal\Core\Datetime\DrupalDateTime;
use Drupal\Core\Entity\ContentEntityForm;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Link;
use Drupal\Core\Url;
use Drupal\nexi_xpay\Plugin\XpayMode\NexiXpayModeManager;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Form controller for the NexiXpayTransaction entity edit forms.
 */
final class NexiXpayTransactionForm extends ContentEntityForm {

  /**
   * The mode manager.
   */
  private NexiXpayModeManager $modeManager;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): self {
    /** @var self $form */
    $form = parent::create($container);
    /** @var NexiXpayModeManager $form['modeManager'] */
    $form->modeManager = $container->get('nexi_xpay.mode_manager');
    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    $form = parent::buildForm($form, $form_state);
    $config = $this->config('nexi_xpay.settings');

    // Hide token fields on creation form (they are auto-generated).
    $is_new = $this->entity->isNew();
    if ($is_new) {
      unset($form['order_id']);
      unset($form['raw_request']);
      unset($form['raw_response']);
      unset($form['last_error']);
    }
    else {
      // Make token fields read-only on edit form.
      foreach (['merchant_reference', 'order_id', 'last_error'] as $field_name) {
        if (isset($form[$field_name]['widget'][0]['value'])) {
          $form[$field_name]['widget'][0]['value']['#disabled'] = TRUE;
        }
      }
      if ($config->get('logging.log_payloads')) {
        foreach (['raw_request', 'raw_response'] as $field_name) {
          if (isset($form[$field_name]['widget'][0]['value'])) {
            $form[$field_name]['widget'][0]['value']['#disabled'] = TRUE;
          }
        }
      }
      else {
        unset($form['raw_request']);
        unset($form['raw_response']);
      }

      // Show pay link on edit form.
      $token = (string) ($this->entity->get('public_token')->value ?? '');
      if ($token !== '') {
        $pay_url = Url::fromRoute('nexi_xpay.pay', [
          'nexi_xpay_transaction' => $this->entity->id(),
        ], [
          'absolute' => TRUE,
          'query' => ['t' => $token],
        ])->toString();

        $link = Link::fromTextAndUrl($pay_url, Url::fromUri($pay_url))->toRenderable();
        $link['#attributes'] = ['target' => '_blank', 'rel' => 'noopener noreferrer'];

        $form['pay_link'] = [
          '#type' => 'item',
          '#title' => $this->t('Pay link'),
          'link' => $link,
          '#weight' => 4.5,
        ];
      }
    }

    // Default token expires
    if ($is_new && isset($form['token_expires']['widget'][0]['value'])) {
      // Prefill only if empty.
      $current = $form['token_expires']['widget'][0]['value']['#default_value'] ?? NULL;
      if (empty($current)) {
        $form['token_expires']['widget'][0]['value']['#default_value'] = DrupalDateTime::createFromTimestamp(\time() + (8 * 60 * 60));
      }
      $form['token_expires']['widget'][0]['value']['#description'] = $this->t('If left empty, it will default to 8 hours from now.');
    }

    // Convert field widget to select using available mode plugins.
    if (isset($form['mode']['widget'][0]['value'])) {
      $options = [];
      foreach ($this->modeManager->getDefinitions() as $plugin_id => $definition) {
        $modePluginLabel = (string) ($definition['label'] ?? $plugin_id);
        $modePluginLabel .= isset($definition['description']) ? ' - ' . $definition['description'] : '';
        $options[$plugin_id] = $modePluginLabel;
      }
      asort($options);

      $form['mode']['widget'][0]['value']['#type'] = 'select';
      $form['mode']['widget'][0]['value']['#options'] = $options;
      $form['mode']['widget'][0]['value']['#empty_option'] = $this->t('- Select a payment mode -');
      $form['mode']['widget'][0]['value']['#description'] = $this->t('Available XPay modes provided by enabled sub-modules.');

      // FORCE dropdown rendering (not listbox).
      $form['mode']['widget'][0]['value']['#size'] = 1;
      $form['mode']['widget'][0]['value']['#multiple'] = FALSE;
      unset($form['mode']['widget'][0]['value']['#attributes']['multiple']);
    }

    return $form;
  }

  /**
   * @inheritDoc
   */
  public function validateForm(array &$form, FormStateInterface $form_state): void {
    parent::validateForm($form, $form_state);

    $mode = (string) $form_state->getValue(['mode', 0, 'value'], '');
    if ($mode === '') {
      $form_state->setErrorByName('mode', $this->t('Please select a payment mode.'));
      return;
    }

    if (!$this->modeManager->hasDefinition($mode)) {
      $form_state->setErrorByName('mode', $this->t('Invalid mode selected.'));
    }
  }

  /**
   * {@inheritdoc}
   */
  public function save(array $form, FormStateInterface $form_state): int {
    $is_new = $this->entity->isNew();
    $status = parent::save($form, $form_state);

    if ($is_new) {
      $token = (string) ($this->entity->get('public_token')->value ?? '');
      if ($token !== '') {
        $pay_url = Url::fromRoute('nexi_xpay.pay', [
          'nexi_xpay_transaction' => $this->entity->id(),
        ], [
          'absolute' => TRUE,
          'query' => ['t' => $token],
        ])->toString();

        $this->messenger()->addStatus($this->t('Pay link: @url', ['@url' => $pay_url]));
      }
    }

    $this->messenger()->addStatus($this->t('Transaction saved.'));
    $form_state->setRedirect('entity.nexi_xpay_transaction.collection');
    return $status;
  }

}
