<?php

declare(strict_types=1);

namespace Drupal\nexi_xpay\Service;

use GuzzleHttp\Exception\GuzzleException;
use JsonException;
use Drupal\Component\Datetime\TimeInterface;
use Drupal\Component\Uuid\UuidInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use GuzzleHttp\ClientInterface;
use Psr\Log\LoggerInterface;

final class NexiXpayClient implements NexiXpayClientInterface {

  public function __construct(
    private readonly ClientInterface $httpClient,
    private readonly ConfigFactoryInterface $configFactory,
    private readonly LoggerInterface $logger,
    private readonly UuidInterface $uuid,
    private readonly TimeInterface $time, // TODO: non viene mai usata, rimuovere?
  ) {}

  public function getEnvironment(): string {
    return (string) $this->configFactory->get('nexi_xpay.settings')->get('environment');
  }

  public function getActiveConfig(): array {
    $cfg = $this->configFactory->get('nexi_xpay.settings');
    $env = $this->getEnvironment();
    return (array) $cfg->get($env);
  }

  public function getBaseUrl(): string {
    $active = $this->getActiveConfig();
    return rtrim((string) ($active['base_url'] ?? ''), '/');
  }

  public function getApiKey(): string {
    $active = $this->getActiveConfig();
    return (string) ($active['api_key'] ?? '');
  }

  public function shouldLogPayloads(): bool {
    return (bool) $this->configFactory->get('nexi_xpay.settings')->get('logging.log_payloads');
  }

  /**
   * Low-level HTTP request helper.
   *
   * Returns: ['status' => int, 'headers' => array, 'body' => string, 'cid' => string]
   * @throws GuzzleException
   */
  public function request(string $method, string $path, array $options = []): array {
    $active = $this->getActiveConfig();
    $baseUrl = rtrim((string) ($active['base_url'] ?? ''), '/');
    $url = $baseUrl . '/' . ltrim($path, '/');

    // IMPORTANT: don't throw on 4xx/5xx; we need body for diagnostics.
    $options += ['http_errors' => FALSE];

    $apiKey = (string) ($active['api_key'] ?? '');
    $correlationId = (string) (($options['headers']['Correlation-Id'] ?? '') ?: $this->uuid->generate());

    $headers = ($options['headers'] ?? []) + [
      // Common:
      'Accept' => 'application/json',
      // Nexi headers:
      'Correlation-Id' => $correlationId,
    ];

    // Defensive: if we're sending JSON, ensure content-type.
    if (isset($options['json']) && empty($headers['Content-Type'])) {
      $headers['Content-Type'] = 'application/json';
    }

    if ($apiKey !== '' && empty($headers['X-Api-Key'])) {
      $headers['X-Api-Key'] = $apiKey;
    }

    $options['headers'] = $headers;

    $this->logger->info('Nexi HTTP {method} {url} cid={cid}', [
      'method' => strtoupper($method),
      'url' => $url,
      'cid' => $correlationId,
    ]);

    if ($this->shouldLogPayloads()) {
      if (isset($options['json'])) {
        $this->logger->debug('HTTP payload json cid={cid}: {json}', [
          'cid' => $correlationId,
          'json' => json_encode($options['json'], JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE),
        ]);
      }
   }

    $resp = $this->httpClient->request($method, $url, $options);

    return [
      'status' => $resp->getStatusCode(),
      'headers' => $resp->getHeaders(),
      'body' => (string) $resp->getBody(),
    ];
  }

  /**
   * JSON helper: decodes body if possible.
   *
   * Returns: ['status'=>int,'data'=>array|null,'raw'=>string,'cid'=>string]
   * @throws GuzzleException
   */
  public function requestJson(string $method, string $path, array $options = []): array {
    $resp = $this->request($method, $path, $options);
    $json = [];

    $body = trim((string) ($resp['body'] ?? ''));
    if ($body !== '') {
      try {
        $decoded = json_decode($body, TRUE, 512, JSON_THROW_ON_ERROR);
        $json = is_array($decoded) ? $decoded : [];
      }
     catch (JsonException $e) {
        // Keep json empty; caller can still use raw body.
        $this->logger->warning('JSON decode failed for {method} {path}: {msg}', [
          'method' => strtoupper($method),
          'path' => $path,
          'msg' => $e->getMessage(),
        ]);
      }
    }

    $resp['json'] = $json;
    return $resp;
  }

  /**
   * Searches for an order and returns its details.
   *
   * For more information about orders, please refer to the dedicated page
   * https://developer.nexi.it/en/api/get-orders-orderId .
   *
   * @throws GuzzleException
   */
  public function getOrder(string $orderId, array $options = []): array {
    $orderId = trim($orderId);
    return $this->requestJson('GET', '/orders/' . rawurlencode($orderId), $options);
  }

}
