# Nexi XPay
**Nexi XPay** is a Drupal 11+ module that enables sites to accept online
payments through Nexi XPay Hosted Payment Page, using standard Drupal
configuration and UI.

The module provides a ready-to-use payment method for site builders,
while also offering a clean and extensible architecture for developers
who need to implement additional Nexi payment modes.

## What this module provides
Out of the box, the module includes:
* Nexi XPay Hosted Payment Page (HPP) integration
* secure redirection to Nexi checkout
* automatic handling of return and cancel callbacks
* server-to-server payment notifications
* reliable and idempotent payment status updates
* user-friendly payment result pages
* a persistent transaction entity with full lifecycle management

No custom code is required to use the default Hosted Payment Page flow.

## Intended audience
This module is designed for two complementary audiences.

### Site builders
Site builders can:
* install and configure the module using the Drupal UI
* manage Nexi credentials and environment settings
* create payment transactions from the admin interface
* share payment links with end users
* rely on Nexi for the hosted checkout experience

No PHP or custom development is required.

### Developers
Developers can:
* extend the module with additional Nexi payment methods
* integrate custom business logic using events
* build alternative UX layers on top of the transaction system
* reuse the provided infrastructure without modifying the core module

## Requirements
* Drupal 11.3 or later
* PHP 8.3 or later
* HTTPS-enabled environment
* Nexi XPay account (required only for production)


## Installation
Install and enable the module in the usual way:
```
composer require drupal/nexi_xpay
drush en nexi_xpay
```
After enabling the module, clear caches.

## Configuration
All configuration is performed through the Drupal administration UI.

* Go to Administration → Configuration → Web Services → Nexi XPay
* Select the environment:
  1. Test (sandbox)
  2. Production
* Enter the required credentials:
  1. Test credentials (provided by Nexi and prefilled)
  2. Production credentials (provided by Nexi)

Production credentials should be treated as confidential and can be overridden
at runtime using `settings.php`. Example:

```php
// settings.php

$config['nexi_xpay.settings']['environment'] = 'production';
$config['nexi_xpay.settings']['production']['api_key'] =
  getenv('NEXI_XPAY_PROD_API_KEY') ?: '';

$config['nexi_xpay.settings']['production']['merchant_id'] =
  getenv('NEXI_XPAY_PROD_MERCHANT_ID') ?: '';

$config['nexi_xpay.settings']['production']['base_url'] =
  getenv('NEXI_XPAY_PROD_BASE_URL') ?: '';

// log payloads for debugging purposes
$config['nexi_xpay.settings']['logging']['log_events'] =
 getenv('NEXI_XPAY_LOG_EVENTS') ?: FALSE;

$config['nexi_xpay.settings']['logging']['log_payloads'] =
  getenv('NEXI_XPAY_LOG_PAYLOADS') ?: FALSE;
```

## Using the Hosted Payment Page
Once configured, the Hosted Payment Page method is immediately available.

To create a payment:

1. Go to Content → Nexi XPay transactions
2. Click Add transaction
3. Fill in the required fields (amount, currency, reference)
4. Save the transaction
5. Use the generated payment link to redirect users to Nexi

Users will be redirected to Nexi’s secure payment page and then
back to the site after completion.

### User experience and theming
The module provides a minimal, neutral user experience:

* a summary page before redirecting to Nexi
* a result page after payment

All pages use standard Drupal theming and can be customized
through the site theme or template overrides.

## Payment result handling
After the payment attempt, users are shown a clear result page indicating:
* Payment successful
* Payment in progress
* Payment failed or canceled

The final payment status is determined automatically based on
Nexi callbacks and order status.

## Notifications and reliability
Nexi sends server-to-server notifications to confirm payment results.

The module:

* validates all notifications securely
* updates transaction status automatically
* safely handles retries and duplicate callbacks
* ensures idempotent state transitions

In environments where notifications are temporarily unavailable
(for example local development), the payment can still be finalized
when the user returns to the site.

## Transaction management
The module defines a dedicated Nexi XPay transaction entity.

Administrators can:

* view the list of transactions
* inspect payment status and metadata
* manage access via Drupal permissions

State transitions are applied consistently and never regressed.

## Permissions
The module integrates with Drupal’s permission system.

Administrators can control:

* access to Nexi configuration pages
* visibility and management of payment transactions

## Non-goals
This module does not:
* provide a shopping cart
* manage products or orders
* perform accounting or invoicing
* replace Nexi’s Hosted Payment Page UI
* use official Nexi PHP SDKs
* handle accounting or fiscal concerns
* enforce a specific user experience

## Developer overview
Developers can:

* implement additional Nexi payment methods
* add new payment flows without altering existing ones
* subscribe to transaction events
* build custom integrations on top of the transaction lifecycle

The Hosted Payment Page implementation included in this module serves
as a reference implementation for additional payment modes.

### Transaction entity
The module defines a dedicated entity representing a Nexi XPay transaction.
State transitions are applied in an idempotent and consistent manner.
To manage the lifecycle of a transaction, go to the entity list page
"Content > Nexi XPay transactions" and you can add/edit/delete transactions.
You can also manage the permissions for each user role to access
the entity list.

### Nexi HTTP client
The module provides a dedicated HTTP client (NexiXpayClient) responsible for:
- performing REST requests to Nexi APIs
- handling JSON responses
- conditional logging of request/response payloads
- retrieving order status via getOrder(orderId)

Application modules should rely on this client rather than using
Guzzle directly.

### Orchestration and idempotency
The module manages:

- application of state transitions only when required
- prevention of invalid or regressive transitions
- event dispatching only when the transaction state actually changes

This ensures safe handling of retries, duplicate notifications,
and repeated callbacks.

### Event system
Transaction-related events are dispatched to allow:
- application-level integrations
- external state synchronization
- custom notifications or logging

Events are emitted only when a real state change occurs.

### Security considerations
The module includes multiple security mechanisms:
- tokenized access to public routes
- strict validation of Nexi notify callbacks
- constant-time comparisons for sensitive values
- optional payload logging
- clear separation between test and production credentials

### Server-to-server notifications (security notes)
The Nexi XPay notification endpoint is designed as a secure,
idempotent webhook and is intentionally not publicly discoverable.

The module generates and sends to Nexi a per-transaction notification
URL with the following format:
```
/nexi-xpay/notify/{transaction_id}/{notify_token}
```
Where:

* `{transaction_id}` is the internal Nexi XPay transaction entity ID.
* `{notify_token}` is a **dedicated, per-transaction secret token**.

This token is **distinct from the public** token used for start/pay flows.

Access control

* The notify route is protected by a custom access check that validates
the notify token against a stored hash.
* Requests without the token, or with an invalid token, do not reach
the controller.
* The endpoint cannot be enumerated using short or predictable URLs.

Response behavior
* The notification endpoint always returns a non-informative response:
204 No Content on success.
* No JSON payload, entity ID, status, or message is ever exposed.

This prevents information disclosure and avoids using the endpoint as an oracle.

Defense in depth in addition to the URL-level token:
* The module still validates the securityToken provided by Nexi inside
the notification payload.
* Transaction status updates are idempotent and monotonic
(final states are never downgraded).

This layered approach ensures safe operation even if notifications
are replayed or received out of order.


### Extending the system
`nexi_xpay` is designed to be extended through:
- separate modules
- payment mode plugins
- event subscribers

New payment methods can be implemented without modifying the module.
For a concrete example, refer to the Hosted Payment Page implementation.

### Create a transaction programmatically
If you need to create a transaction from custom code (for example from a custom
form submit, a controller, or a service), you can create a
`nexi_xpay_transaction` entity using the Entity API:

```php
use Drupal\nexi_xpay\Entity\NexiXpayTransactionInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;

/** @var \Drupal\Core\Entity\EntityStorageInterface $storage */
$storage = \Drupal::entityTypeManager()->getStorage('nexi_xpay_transaction');

// Create a new transaction.
// Amount is stored in minor units (e.g. 1000 = €10.00).
$transaction = $storage->create([
  'merchant_reference' => 'ORDER-2026-0001',
  'amount' => 1000,
  'currency' => 'EUR',
  'mode' => 'hpp_redirect',
]);

$transaction->save();
```

Recommended practices:
* Use a unique `merchant_reference` (it is used as Nexi `orderId`).
* Keep `amount` in minor units (cents).
* Set `currency` because it is mandatory for Nexi.
* Set `mode` to the payment mode plugin ID (HPP included in the module).

Use this anywhere in custom code (controller, service, form submit, etc.):

```php
/**
 * Creates a Nexi XPay transaction programmatically.
 *
 * Amount is in minor units (e.g. 1000 = €10.00).
 */
function my_module_create_nexi_transaction(): int {
  /** @var \Drupal\Core\Entity\EntityStorageInterface $storage */
  $storage = \Drupal::entityTypeManager()->getStorage('nexi_xpay_transaction');

  /** @var \Drupal\Core\Entity\ContentEntityInterface $transaction */
  $transaction = $storage->create([
    // Optional but recommended: your internal reference.
    'merchant_reference' => 'ORDER-2026-0001',

    // Required:
    'amount' => 1000,
    'currency' => 'EUR',

    // Required: payment mode plugin id (HPP included in the module).
    'mode' => 'hpp_redirect',

    // Optional: status defaults to 'pending' if omitted.
    // 'status' => 'pending',.
  ]);

  // On first save, the module auto-generates:
  // - public_token (plain)
  // - public_token_hash (sha256)
  // - token_expires (now + 8h, if not provided)
  $transaction->save();

  return (int) $transaction->id();
}
```

### Get the Pay URL programmatically
Once you have a transaction, you can generate the Pay URL (including the
public token parameter) and redirect the user to start the payment.

Generate the URL:
```php
use Drupal\Core\Url;

/**
 * @var \Drupal\nexi_xpay\Entity\NexiXpayTransactionInterface $transaction */

// Ensure the transaction has a public token.
// Depending on your implementation, the token may be generated on save,
// or you may have a dedicated method to ensure it exists.
$token = $transaction->get('public_token')->value;

// Build an absolute URL to the pay route with the token as query parameter.
$url = Url::fromRoute('nexi_xpay.pay', [
  'transaction' => $transaction->id(),
], [
  'absolute' => TRUE,
  'query' => ['t' => $token],
]);

$pay_url = $url->toString();

```
Redirect to the Pay URL (controller example):
```php
use Symfony\Component\HttpFoundation\RedirectResponse;

return new RedirectResponse($pay_url);

```

Notes:
* The Pay URL is token-protected via the `t` query parameter.
* Use `absolute => TRUE` when generating links for emails or external contexts.

This generates the tokenized pay link
(`/nexi-xpay/pay/{nexi_xpay_transaction}?t=...`)
exactly as the module expects:

```php

use Drupal\Core\Url;

/**
 * Builds an absolute Pay URL for a Nexi XPay transaction.
 */
function my_module_get_nexi_pay_url(int $transaction_id): string {
  /** @var \Drupal\Core\Entity\EntityStorageInterface $storage */
  $storage = \Drupal::entityTypeManager()->getStorage('nexi_xpay_transaction');

  /** @var \Drupal\Core\Entity\ContentEntityInterface|null $transaction */
  $transaction = $storage->load($transaction_id);
  if (!$transaction) {
    throw new \InvalidArgumentException(
      "Transaction {$transaction_id} not found."
    );
  }

  // Token is stored in plain and validated via hash on access check.
  $token = (string) ($transaction->get('public_token')->value ?? '');
  if ($token === '') {
    // In normal usage the token is generated on first save.
    // If you are loading legacy data, ensure it is saved at least once.
    $transaction->save();
    $token = (string) ($transaction->get('public_token')->value ?? '');
  }

  $url = Url::fromRoute('nexi_xpay.pay', [
    'nexi_xpay_transaction' => $transaction->id(),
  ], [
    'absolute' => TRUE,
    'query' => ['t' => $token],
  ]);

  return $url->toString();
}

```
Optional: redirect the user to the Pay URL (controller example):
```php

use Symfony\Component\HttpFoundation\RedirectResponse;

$pay_url = my_module_get_nexi_pay_url($transaction_id);
return new RedirectResponse($pay_url);
```

### Start the payment directly (skip the summary page)
In some cases you may want to start the payment immediately,
without showing the intermediate summary page.
The module exposes a dedicated start route for this purpose.

Generate the Direct Start URL
```php
use Drupal\Core\Url;

/**
 * Builds an absolute Direct Start URL for a Nexi XPay transaction.
 */
function my_module_get_nexi_start_url(int $transaction_id): string {
  /** @var \Drupal\Core\Entity\EntityStorageInterface $storage */
  $storage = \Drupal::entityTypeManager()->getStorage('nexi_xpay_transaction');

  /** @var \Drupal\Core\Entity\ContentEntityInterface|null $transaction */
  $transaction = $storage->load($transaction_id);
  if (!$transaction) {
    throw new \InvalidArgumentException(
      "Transaction {$transaction_id} not found."
    );
  }

  // Ensure the public token exists.
  $token = (string) ($transaction->get('public_token')->value ?? '');
  if ($token === '') {
    $transaction->save();
    $token = (string) ($transaction->get('public_token')->value ?? '');
  }

  $url = Url::fromRoute('nexi_xpay.start', [
    'nexi_xpay_transaction' => $transaction->id(),
  ], [
    'absolute' => TRUE,
    'query' => ['t' => $token],
  ]);

  return $url->toString();
}

```

Redirect to the Direct Start URL
```php
use Symfony\Component\HttpFoundation\RedirectResponse;

$start_url = my_module_get_nexi_start_url($transaction_id);
return new RedirectResponse($start_url);
```
Notes:
* The direct start route is token-protected, just like the Pay route.
* This flow is useful when:
  - the transaction details are already known
  - no user confirmation page is required
* The standard return and notify handling is unchanged.
