<?php

declare(strict_types=1);

namespace Drupal\nexi_xpay\Access;

use Drupal\Core\Access\AccessResult;
use Drupal\Core\Session\AccountInterface;
use Drupal\nexi_xpay\Entity\NexiXpayTransactionInterface;
use Symfony\Component\HttpFoundation\Request;

/**
 * Defines a class for performing access checks based on a transaction token.
 *
 * This class provides a static method for verifying if an incoming request
 * contains a valid token. The check involves verifying the presence
 * of the token, matching it against the expected hashed value,
 * and ensuring it has not expired.
 *
 *
 * Usage of this class ensures secure access control to resources linked
 * to a transaction in systems like Nexi XPay.
 */
final class TransactionTokenAccessCheck {

  /**
   * Custom access callback for the transaction route.
   */
  public static function access(AccountInterface $account, NexiXpayTransactionInterface $nexi_xpay_transaction, Request $request): AccessResult {
    $token = (string) $request->query->get('t', '');
    if ($token === '') {
      return AccessResult::forbidden('Missing token.')->setCacheMaxAge(0);
    }

    $hash = $nexi_xpay_transaction->getPublicTokenHash();
    if (!$hash) {
      return AccessResult::forbidden('Token not configured.')->setCacheMaxAge(0);
    }

    $expires = $nexi_xpay_transaction->getTokenExpiresTime();
    if ($expires !== NULL && \time() > $expires) {
      return AccessResult::forbidden('Token expired.')->setCacheMaxAge(0);
    }

    $calc = hash('sha256', $token);
    if (!hash_equals($hash, $calc)) {
      return AccessResult::forbidden('Invalid token.')->setCacheMaxAge(0);
    }

    return AccessResult::allowed()->setCacheMaxAge(0);
  }

}
