<?php

declare(strict_types=1);

namespace Drupal\nexi_xpay\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\nexi_xpay\Entity\NexiXpayTransactionInterface;
use Drupal\nexi_xpay\Service\NexiXpayManagerInterface;
use Drupal\nexi_xpay\Service\TransactionStatusUpdater;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\HttpKernel\Exception\NotFoundHttpException;

/**
 * Implements the controller for the payment process.
 */
final class PaymentController extends ControllerBase {

  public function __construct(
    private readonly NexiXpayManagerInterface $manager,
    private readonly TransactionStatusUpdater $statusUpdater,
  ) {}

  /**
   * Inject dependencies.
   *
   * @param \Symfony\Component\DependencyInjection\ContainerInterface $container
   *   The container.
   *
   * @return PaymentController
   *   The controller.
   */
  public static function create(ContainerInterface $container): self {
    /** @var \Drupal\nexi_xpay\Service\NexiXpayManagerInterface $manager */
    $manager = $container->get('nexi_xpay.manager');

    /** @var \Drupal\nexi_xpay\Service\TransactionStatusUpdater $statusUpdater */
    $statusUpdater = $container->get('nexi_xpay.status_updater');

    return new self($manager, $statusUpdater);
  }

  /**
   * Start payment.
   *
   * @param \Drupal\nexi_xpay\Entity\NexiXpayTransactionInterface $nexi_xpay_transaction
   *   The transaction entity.
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The current request.
   *
   * @return array<string, mixed>|Response
   *   - array: render array for the payment page.
   *   - Response: redirect to the payment page.
   */
  public function start(NexiXpayTransactionInterface $nexi_xpay_transaction, Request $request): array|Response {
    $result = $this->manager->startPayment($nexi_xpay_transaction, $request);
    if ($result->response) {
      return $result->response;
    }
    // Fallback: if a mode returns render by mistake, render it.
    return $result->renderArray ?? ['#markup' => 'Unable to start payment.'];
  }

  /**
   * Pay page.
   *
   * @param \Drupal\nexi_xpay\Entity\NexiXpayTransactionInterface $nexi_xpay_transaction
   *   The transaction entity.
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The current request.
   *
   * @return array<string, mixed>
   *   The render array for the pay page.
   */
  public function pay(NexiXpayTransactionInterface $nexi_xpay_transaction, Request $request): array {
    return $this->manager->buildPayPage($nexi_xpay_transaction, $request);
  }

  /**
   * Handle return by order.
   *
   * @param string $orderId
   *   The order ID.
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The current request.
   *
   * @return array<string, mixed>
   *   The render array for the return page.
   *
   * @throws \Drupal\Core\Entity\EntityStorageException
   * @throws \Drupal\Component\Plugin\Exception\InvalidPluginDefinitionException
   * @throws \Drupal\Component\Plugin\Exception\PluginNotFoundException
   */
  public function handleReturnByOrder(string $orderId, Request $request): array {
    $tx = $this->entityTypeManager()
      ->getStorage('nexi_xpay_transaction')
      ->loadByProperties(['order_id' => $orderId]);

    $tx = $tx ? reset($tx) : NULL;
    if (!$tx instanceof NexiXpayTransactionInterface) {
      throw new NotFoundHttpException();
    }

    // Reuse existing return handler logic (handle + render via plugin).
    return $this->handleReturn($tx, $request);
  }

  /**
   * Handle cancel by order.
   *
   * @param string $orderId
   *   The order ID.
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The current request.
   *
   * @return array<string, mixed>
   *   The render array for the return page.
   *
   * @throws \Drupal\Core\Entity\EntityStorageException
   * @throws \Drupal\Component\Plugin\Exception\InvalidPluginDefinitionException
   * @throws \Drupal\Component\Plugin\Exception\PluginNotFoundException
   */
  public function handleCancelByOrder(string $orderId, Request $request): array {
    $tx = $this->entityTypeManager()
      ->getStorage('nexi_xpay_transaction')
      ->loadByProperties(['order_id' => $orderId]);

    $tx = $tx ? reset($tx) : NULL;
    if (!$tx instanceof NexiXpayTransactionInterface) {
      throw new NotFoundHttpException();
    }

    return $this->handleReturn($tx, $request);
  }

  /**
   * Handle Xpay return.
   *
   * @param \Drupal\nexi_xpay\Entity\NexiXpayTransactionInterface $nexi_xpay_transaction
   *   The transaction entity.
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The current request.
   *
   * @return array<string, mixed>
   *   The render array for the return page.
   *
   * @throws \Drupal\Core\Entity\EntityStorageException
   */
  public function handleReturn(NexiXpayTransactionInterface $nexi_xpay_transaction, Request $request): array {
    $handle = $this->manager->handleReturn($nexi_xpay_transaction, $request);

    if ($handle->newStatus) {
      $this->statusUpdater->setStatusIfChanged($nexi_xpay_transaction, $handle->newStatus, $handle->context);
    }

    return $this->manager->buildReturnPage($nexi_xpay_transaction, $request, $handle);
  }

  /**
   * Handle Xpay notification.
   *
   * @param \Drupal\nexi_xpay\Entity\NexiXpayTransactionInterface $nexi_xpay_transaction
   *   The transaction entity.
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The current request.
   *
   * @return \Symfony\Component\HttpFoundation\Response
   *   The response to send back to Nexi.
   *
   * @throws \Drupal\Core\Entity\EntityStorageException
   */
  public function handleNotify(NexiXpayTransactionInterface $nexi_xpay_transaction, Request $request): Response {
    $res = $this->manager->handleNotify($nexi_xpay_transaction, $request);

    if ($res->newStatus) {
      $this->statusUpdater->setStatusIfChanged($nexi_xpay_transaction, $res->newStatus, $res->context);
    }

    // Idempotent webhook endpoint:
    // - Do not leak entity id/status/message.
    // - Keep response constant to avoid information disclosure.
    return new Response('', 204);
  }

}
