<?php

declare(strict_types=1);

namespace Drupal\nexi_xpay\Entity;

use Drupal\Core\Entity\Attribute\ContentEntityType;
use Drupal\Core\Entity\ContentEntityBase;
use Drupal\Core\Entity\EntityTypeInterface;
use Drupal\Core\Field\BaseFieldDefinition;
use Drupal\Core\Entity\EntityStorageInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\nexi_xpay\Util\SecretGenerator;
use Drupal\nexi_xpay\Util\TypeCast;

/**
 * {@inheritdoc}
 */
#[ContentEntityType(
  id: 'nexi_xpay_transaction',
  label: new TranslatableMarkup('Nexi Xpay Transaction'),
  entity_keys: [
    'id' => 'id',
    'uuid' => 'uuid',
  ],
  handlers: [
    'list_builder' => 'Drupal\nexi_xpay\ListBuilder\NexiXpayTransactionListBuilder',
    'form' => [
      'add' => 'Drupal\nexi_xpay\Form\NexiXpayTransactionForm',
      'edit' => 'Drupal\nexi_xpay\Form\NexiXpayTransactionForm',
      'delete' => 'Drupal\Core\Entity\ContentEntityDeleteForm',
    ],
    'access' => 'Drupal\Core\Entity\EntityAccessControlHandler',
  ],
  links: [
    'collection' => '/admin/content/nexi-xpay/transactions',
    'add-form' => '/admin/content/nexi-xpay/transactions/add',
    'edit-form' => '/admin/content/nexi-xpay/transactions/{nexi_xpay_transaction}/edit',
    'delete-form' => '/admin/content/nexi-xpay/transactions/{nexi_xpay_transaction}/delete',
  ],
  admin_permission: 'administer nexi-xpay transactions',
  base_table: 'nexi_xpay_transaction'
)]
final class NexiXpayTransaction extends ContentEntityBase implements NexiXpayTransactionInterface {

  use StringTranslationTrait;

  /**
   * {@inheritdoc}
   *
   * @return array<string, \Drupal\Core\Field\BaseFieldDefinition>
   *   An array of base field definitions for the entity type, keyed by field
   *   name.
   */
  public static function baseFieldDefinitions(EntityTypeInterface $entity_type): array {
    /** @var array<string, \Drupal\Core\Field\BaseFieldDefinition> $fields */
    $fields = parent::baseFieldDefinitions($entity_type);

    $fields['uuid']->setLabel(new TranslatableMarkup('UUID'));

    $fields['status'] = BaseFieldDefinition::create('list_string')
      ->setLabel(new TranslatableMarkup('Status'))
      ->setRequired(TRUE)
      ->setSettings(['allowed_values' => self::getStates()])
      ->setDefaultValue(self::getDefaultStatus())
      ->setDisplayOptions('form', ['type' => 'options_select', 'weight' => 0])
      ->setDisplayConfigurable('form', TRUE);

    $fields['merchant_reference'] = BaseFieldDefinition::create('string')
      ->setLabel(new TranslatableMarkup('Merchant reference'))
      ->setDescription(new TranslatableMarkup('Reference with the merchant system business logic. Auto-generated if empty.'))
      ->setSetting('max_length', 255)
      ->setDisplayOptions('form', ['type' => 'string_textfield', 'weight' => 1])
      ->setDisplayConfigurable('form', TRUE);

    $fields['amount'] = BaseFieldDefinition::create('integer')
      ->setLabel(new TranslatableMarkup('Amount (minor units)'))
      ->setRequired(TRUE)
      ->setDisplayOptions('form', ['type' => 'number', 'weight' => 2])
      ->setDisplayConfigurable('form', TRUE);

    $fields['currency'] = BaseFieldDefinition::create('string')
      ->setLabel(new TranslatableMarkup('Currency'))
      ->setRequired(TRUE)
      ->setSetting('max_length', 3)
      ->setDefaultValue('EUR')
      ->setDisplayOptions('form', ['type' => 'string_textfield', 'weight' => 3])
      ->setDisplayConfigurable('form', TRUE);

    $fields['mode'] = BaseFieldDefinition::create('string')
      ->setLabel(new TranslatableMarkup('Mode'))
      ->setDescription(new TranslatableMarkup('Payment mode plugin id (selected from enabled mode plugins).'))
      ->setRequired(TRUE)
      ->setSetting('max_length', 64)
      ->setDisplayOptions('form', [
        'type' => 'options_select',
        'weight' => 4,
      ])
      ->setDisplayConfigurable('form', TRUE);

    $fields['public_token'] = BaseFieldDefinition::create('string')
      ->setLabel(new TranslatableMarkup('Public token (plain)'))
      ->setDescription(new TranslatableMarkup('Cleartext token used to generate the public link. Stored for admin use.'))
      ->setSetting('max_length', 64);

    $fields['public_token_hash'] = BaseFieldDefinition::create('string')
      ->setLabel(new TranslatableMarkup('Public token hash'))
      ->setSetting('max_length', 128)
      ->setDescription(new TranslatableMarkup('Store only the hash (SHA-256 hex).'));

    $fields['token_expires'] = BaseFieldDefinition::create('timestamp')
      ->setLabel(new TranslatableMarkup('Token expires'))
      ->setDescription(new TranslatableMarkup('Optional public token expiry timestamp.'))
      ->setDisplayOptions('form', ['type' => 'datetime_timestamp', 'weight' => 6])
      ->setDisplayConfigurable('form', TRUE);

    $fields['notify_token'] = BaseFieldDefinition::create('string')
      ->setLabel(new TranslatableMarkup('Notify token (plain)'))
      ->setDescription(new TranslatableMarkup('Token used in notificationUrl (admin/debug use).'))
      ->setSetting('max_length', 64);

    $fields['notify_token_hash'] = BaseFieldDefinition::create('string')
      ->setLabel(new TranslatableMarkup('Notify token hash'))
      ->setSetting('max_length', 128)
      ->setDescription(new TranslatableMarkup('Store only the hash (SHA-256 hex) for notify access check.'));

    // Nexi correlation fields.
    $fields['order_id'] = BaseFieldDefinition::create('string')
      ->setLabel(new TranslatableMarkup('NEXI Order ID Reference'))
      ->setDescription(new TranslatableMarkup('Max 18, lowercase [a-z0-9]. Auto-generated from merchant_reference if empty.'))
      ->setSetting('max_length', 18)
      ->setDisplayOptions('form', ['type' => 'string_textfield', 'weight' => 7])
      ->setDisplayConfigurable('form', TRUE);

    $fields['session_id'] = BaseFieldDefinition::create('string')
      ->setLabel(new TranslatableMarkup('Session ID'))
      ->setSetting('max_length', 128);

    $fields['security_token'] = BaseFieldDefinition::create('string')
      ->setLabel(new TranslatableMarkup('Security token'))
      ->setSetting('max_length', 128)
      ->setDescription(new TranslatableMarkup('Used to validate notify callbacks (mode-specific).'));

    $fields['created'] = BaseFieldDefinition::create('created')->setLabel(new TranslatableMarkup('Created'));
    $fields['changed'] = BaseFieldDefinition::create('changed')->setLabel(new TranslatableMarkup('Changed'));

    return $fields;
  }

  /**
   * {@inheritDoc}
   */
  public function preSave(EntityStorageInterface $storage): void {
    parent::preSave($storage);

    // Default status: pending if empty.
    if ($this->get('status')->isEmpty()) {
      $this->set('status', $this->getDefaultStatus());
    }

    if ($this->isNew()) {
      // Generate a Nexi-compliant id.
      $generated_order_id = SecretGenerator::generateBase36Id(18);

      if ($this->get('order_id')->isEmpty() || $this->get('order_id')->value === '') {
        $this->set('order_id', $generated_order_id);
      }

      // Auto-generate merchant_reference if empty: max 18, [a-z0-9].
      // In this case merchant_reference will be equal to order_id.
      if ($this->get('merchant_reference')->isEmpty() || $this->get('merchant_reference')->value === '') {
        $this->set('merchant_reference', $generated_order_id);
      }

      // Auto-generate public token and hash if missing.
      $plain = TypeCast::toString($this->get('public_token')->value ?? '');
      $hash = TypeCast::toString($this->get('public_token_hash')->value ?? '');

      if ($plain === '') {
        $generated_public_token = SecretGenerator::generateHexSecretWithHash(16);
        $this->set('public_token', $generated_public_token['secret']);
        $this->set('public_token_hash', $generated_public_token['hash']);
      }
      elseif ($hash === '') {
        $this->set('public_token_hash', hash('sha256', $plain));
      }

      // Default token expiry: now + 8 hours (creation only) if not provided.
      $expires = TypeCast::toInt($this->get('token_expires')->value);

      // Treat empty / 0 as not set.
      if (empty($expires) || $expires <= 0) {
        $this->set('token_expires', \time() + (8 * 60 * 60));
      }

      // Auto-generate notify token only on first creation if missing.
      // Token in plain (admin/debug use) and hash for notify access check.
      $notify_plain = TypeCast::toString($this->get('notify_token')->value ?? '');
      $notify_hash = TypeCast::toString($this->get('notify_token_hash')->value ?? '');

      if ($notify_plain === '') {
        $generated_notify_token = SecretGenerator::generateHexSecretWithHash(32);
        $this->set('notify_token', $generated_notify_token['secret']);
        $this->set('notify_token_hash', $generated_notify_token['hash']);
      }
      elseif ($notify_hash === '') {
        $this->set('notify_token_hash', hash('sha256', $notify_plain));
      }
    }

  }

  /**
   * {@inheritDoc}
   */
  public static function getStates(): array {
    return [
      self::STATUS_PENDING => new TranslatableMarkup('Pending'),
      self::STATUS_PROCESSING => new TranslatableMarkup('Processing'),
      self::STATUS_PAID => new TranslatableMarkup('Paid'),
      self::STATUS_FAILED => new TranslatableMarkup('Failed'),
      self::STATUS_CANCELLED => new TranslatableMarkup('Cancelled'),
      self::STATUS_EXPIRED => new TranslatableMarkup('Expired'),
    ];
  }

  /**
   * {@inheritDoc}
   */
  public static function getDefaultStatus(): string {
    return self::STATUS_PENDING;
  }

  /**
   * {@inheritDoc}
   */
  public static function getFinalStatuses(): array {
    return self::FINAL_STATUSES;
  }

  /**
   * {@inheritDoc}
   */
  public function setStatus(string $status): static {
    $this->set('status', $status);
    return $this;
  }

  /**
   * {@inheritDoc}
   */
  public function getStatus(): string {
    return TypeCast::toString($this->get('status')->value, self::getDefaultStatus());
  }

  /**
   * {@inheritDoc}
   */
  public function isInFinalStatus(): bool {
    return in_array(
      $this->getStatus(),
      NexiXpayTransactionInterface::FINAL_STATUSES,
      TRUE
    );
  }

  /**
   * {@inheritDoc}
   */
  public function getMerchantReference(): string {
    return TypeCast::toString($this->get('merchant_reference')->value);
  }

  /**
   * {@inheritDoc}
   */
  public function getAmount(): int {
    return TypeCast::toInt($this->get('amount')->value);
  }

  /**
   * {@inheritDoc}
   */
  public function getAmountFormatted(): string {
    return number_format($this->getAmount() / 100, 2, '.', '');
  }

  /**
   * {@inheritDoc}
   */
  public static function getDefaultCurrency(): string {
    return 'EUR';
  }

  /**
   * {@inheritDoc}
   */
  public function getCurrency(): string {
    return TypeCast::toString($this->get('currency')->value, self::getDefaultCurrency());
  }

  /**
   * {@inheritDoc}
   */
  public function setMode(string $mode): static {
    $this->set('mode', $mode);
    return $this;
  }

  /**
   * {@inheritDoc}
   */
  public function getMode(): string {
    return TypeCast::toString($this->get('mode')->value, 'hpp_redirect');
  }

  /**
   * {@inheritDoc}
   */
  public function getPublicToken(): string {
    return TypeCast::toString($this->get('public_token')->value);
  }

  /**
   * {@inheritDoc}
   */
  public function setPublicTokenHash(string $hash): static {
    $this->set('public_token_hash', $hash);
    return $this;
  }

  /**
   * {@inheritDoc}
   */
  public function getPublicTokenHash(): ?string {
    $v = TypeCast::toString($this->get('public_token_hash')->value);
    return ($v === '') ? NULL : $v;
  }

  /**
   * {@inheritDoc}
   */
  public function setTokenExpiresTime(?int $timestamp): static {
    $this->set('token_expires', $timestamp);
    return $this;
  }

  /**
   * {@inheritDoc}
   */
  public function getTokenExpiresTime(): ?int {
    $v = TypeCast::toInt($this->get('token_expires')->value);
    return ($v === 0) ? NULL : $v;
  }

  /**
   * {@inheritDoc}
   */
  public function setNotifyToken(string $token): static {
    $this->set('notify_token', $token);
    return $this;
  }

  /**
   * {@inheritDoc}
   */
  public function getNotifyToken(): string {
    return TypeCast::toString($this->get('notify_token')->value);
  }

  /**
   * {@inheritDoc}
   */
  public function setNotifyTokenHash(string $hash): static {
    $this->set('notify_token_hash', $hash);
    return $this;
  }

  /**
   * {@inheritDoc}
   */
  public function getNotifyTokenHash(): ?string {
    $v = TypeCast::toString($this->get('notify_token_hash')->value);
    return ($v === '') ? NULL : $v;
  }

  /**
   * {@inheritDoc}
   */
  public function getOrderId(): string {
    return TypeCast::toString($this->get('order_id')->value);
  }

  /**
   * {@inheritDoc}
   */
  public function getSecurityToken(): string {
    return TypeCast::toString($this->get('security_token')->value);
  }

  /**
   * {@inheritdoc}
   */
  public function getChangedTime(): int {
    return TypeCast::toInt($this->get('changed')->value);
  }

  /**
   * {@inheritdoc}
   */
  public function setChangedTime($timestamp): static {
    $this->set('changed', (int) $timestamp);
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getChangedTimeAcrossTranslations(): int {
    // For this MVP we do not manage translations; return the base changed time.
    // Drupal core expects an int.
    return $this->getChangedTime();
  }

}
