<?php

declare(strict_types=1);

namespace Drupal\nexi_xpay\Entity;

use Drupal\Core\Entity\ContentEntityInterface;
use Drupal\Core\Entity\EntityChangedInterface;

/**
 * The Nexi Xpay Transaction entity.
 *
 * This content entity is used to manage transactions processed
 * via the Nexi Xpay payment gateway.
 *
 * Key features:
 * - Auto-generation of merchant references if none is provided.
 * - Auto-generation of public tokens and their corresponding hashes.
 * - Auto-generation of notify tokens and their corresponding hashes.
 * - Auto-generation of security tokens.
 * - Public token expiry management.
 * - Tracks transaction states, amounts, and payment modes.
 */
interface NexiXpayTransactionInterface extends ContentEntityInterface, EntityChangedInterface {

  /**
   * Status constants.
   */
  public const string STATUS_PENDING = 'pending';
  public const string STATUS_PROCESSING = 'processing';
  public const string STATUS_PAID = 'paid';
  public const string STATUS_FAILED = 'failed';
  public const string STATUS_CANCELLED = 'cancelled';
  public const string STATUS_EXPIRED = 'expired';

  /**
   * Refunded status.
   */
  public const string STATUS_REFUNDED = 'refunded';

  /**
   * Transaction final statuses.
   */
  public const array FINAL_STATUSES = [
    self::STATUS_PAID,
    self::STATUS_FAILED,
    self::STATUS_CANCELLED,
    self::STATUS_EXPIRED,
  ];

  /**
   * Returns the NexiXpayTransaction states.
   *
   * @return array<string, \Drupal\Core\StringTranslation\TranslatableMarkup>
   *   The available states.
   */
  public static function getStates(): array;

  /**
   * Returns the default transaction status.
   *
   * @return string
   *   The default transaction status.
   */
  public static function getDefaultStatus(): string;

  /**
   * Returns the final transaction statuses.
   *
   * @return array<string>
   *   The final transaction statuses.
   */
  public static function getFinalStatuses(): array;

  /**
   * Sets the NexiXpayTransaction status.
   *
   * @param string $status
   *   The transaction status.
   *
   * @return static
   */
  public function setStatus(string $status): static;

  /**
   * Returns the transaction status.
   *
   * @return string
   *   The transaction status.
   */
  public function getStatus(): string;

  /**
   * Returns TRUE if the transaction is already finalized.
   *
   * @return bool
   *   The results.
   */
  public function isInFinalStatus(): bool;

  /**
   * Returns the merchant reference.
   *
   * @return string
   *   Reference provided by the merchant.
   */
  public function getMerchantReference(): string;

  /**
   * Returns the transaction amount in minor units.
   *
   * @return int
   *   The transaction amount in minor units, e.g. cents.
   */
  public function getAmount(): int;

  /**
   * Returns the transaction amount formatted for display.
   *
   * @return string
   *   Return formatted amount.
   */
  public function getAmountFormatted(): string;

  /**
   * Returns the default transaction currency.
   *
   * @return string
   *   The default currency code, e.g. EUR.
   */
  public static function getDefaultCurrency(): string;

  /**
   * Returns the transaction currency.
   *
   * @return string
   *   The currency code, e.g. EUR.
   */
  public function getCurrency(): string;

  /**
   * Sets the payment mode identifier.
   *
   * @param string $mode
   *   The payment mode plugin used for this transaction.
   *
   * @return static
   */
  public function setMode(string $mode): static;

  /**
   * Returns the payment mode identifier.
   *
   * @return string
   *   The payment mode plugin used for this transaction.
   */
  public function getMode(): string;

  /**
   * Returns the public token in plain form (admin/debug use).
   *
   * @return string
   *   The public token of transaction.
   */
  public function getPublicToken(): string;

  /**
   * Sets the public token hash.
   *
   * @param string $hash
   *   The token hash, SHA-256 hex.
   *
   * @return static
   */
  public function setPublicTokenHash(string $hash): static;

  /**
   * Returns the public token.
   *
   * @return string|null
   *   The token of transaction.
   */
  public function getPublicTokenHash(): ?string;

  /**
   * Sets the public token expiry timestamp.
   *
   * @param int|null $timestamp
   *   Expiry timestamp, or NULL to clear the expiry.
   *
   * @return static
   */
  public function setTokenExpiresTime(?int $timestamp): static;

  /**
   * Returns the public token expiry timestamp.
   *
   * @return int|null
   *   The expiry timestamp, or NULL if the token never expires.
   */
  public function getTokenExpiresTime(): ?int;

  /**
   * Sets the notification token in plain form (admin/debug use).
   */
  public function setNotifyToken(string $token): static;

  /**
   * Returns the notification token in plain form (admin/debug use).
   *
   * This value is used to build the notificationUrl sent to Nexi.
   */
  public function getNotifyToken(): string;

  /**
   * Sets the notification token hash (SHA-256 hex).
   */
  public function setNotifyTokenHash(string $hash): static;

  /**
   * Returns the notification token hash (SHA-256 hex).
   */
  public function getNotifyTokenHash(): ?string;

  /**
   * Returns the NEXI Order ID.
   *
   * @return string
   *   Nexi Order ID.
   */
  public function getOrderId(): string;

  /**
   * Returns the security token.
   *
   * @return string
   *   Security token.
   */
  public function getSecurityToken(): string;

}
