<?php

declare(strict_types=1);

namespace Drupal\nexi_xpay\Form;

use Drupal\Core\Form\ConfirmFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\Core\Url;
use Drupal\nexi_xpay\Entity\NexiXpayTransactionInterface;
use Drupal\nexi_xpay\Util\SecretGenerator;
use Symfony\Component\HttpKernel\Exception\NotFoundHttpException;

/**
 * Form controller for the regenerate token action.
 */
final class RegenerateTokenForm extends ConfirmFormBase {

  /**
   * The transaction entity.
   */
  private NexiXpayTransactionInterface $transaction;

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'nexi_xpay_regenerate_token_form';
  }

  /**
   * {@inheritDoc}
   *
   * @param array<string, mixed> $form
   *   An associative array containing the structure of the form.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The current state of the form.
   * @param \Drupal\nexi_xpay\Entity\NexiXpayTransactionInterface|null $nexi_xpay_transaction
   *   Transaction entity.
   *
   * @return array<string, mixed>
   *   The form structure.
   */
  public function buildForm(
    array $form,
    FormStateInterface $form_state,
    ?NexiXpayTransactionInterface $nexi_xpay_transaction = NULL,
  ): array {
    if (!$nexi_xpay_transaction) {
      throw new NotFoundHttpException();
    }

    $this->transaction = $nexi_xpay_transaction;
    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritDoc}
   */
  public function getQuestion(): TranslatableMarkup {
    return $this->t('Regenerate public token for transaction @id?', ['@id' => $this->transaction->id()]);
  }

  /**
   * {@inheritDoc}
   */
  public function getDescription(): TranslatableMarkup {
    return $this->t('This will invalidate the previous public payment link.');
  }

  /**
   * {@inheritDoc}
   */
  public function getCancelUrl(): Url {
    return Url::fromRoute('entity.nexi_xpay_transaction.collection');
  }

  /**
   * {@inheritDoc}
   */
  public function getConfirmText(): TranslatableMarkup {
    return $this->t('Regenerate token');
  }

  /**
   * {@inheritdoc}
   *
   * @param array<string, mixed> $form
   *   An associative array containing the structure of the form.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The current state of the form.
   *
   * @throws \Random\RandomException|\Drupal\Core\Entity\EntityStorageException
   *   Catches all exceptions and returns a generic error message.
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    // Regenerate token is allowed only when the transaction is pending.
    if ($this->transaction->getStatus() !== NexiXpayTransactionInterface::STATUS_PENDING) {
      $this->messenger()->addError($this->t('Token can be regenerated only when the transaction is pending.'));
      $form_state->setRedirectUrl($this->getCancelUrl());
      return;
    }

    $plain = SecretGenerator::generateHexSecretWithHash(16);
    $this->transaction->set('public_token', $plain['secret']);
    $this->transaction->set('public_token_hash', $plain['hash']);
    $this->transaction->save();

    $pay_url = Url::fromRoute('nexi_xpay.pay', [
      'nexi_xpay_transaction' => $this->transaction->id(),
    ], [
      'absolute' => TRUE,
      'query' => ['t' => $plain['secret']],
    ])->toString();

    $this->messenger()->addStatus($this->t('Token regenerated. New pay link: @url', ['@url' => $pay_url]));
    $form_state->setRedirectUrl($this->getCancelUrl());
  }

}
