<?php

declare(strict_types=1);

namespace Drupal\nexi_xpay\ListBuilder;

use Drupal\Core\Datetime\DateFormatterInterface;
use Drupal\Core\Entity\EntityHandlerInterface;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Entity\EntityListBuilder;
use Drupal\Core\Link;
use Drupal\Core\Url;
use Drupal\nexi_xpay\Entity\NexiXpayTransactionInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Defines a class to build a listing of NexiXpayTransaction entities.
 */
final class NexiXpayTransactionListBuilder extends EntityListBuilder {

  /**
   * The date formatter service.
   */
  protected DateFormatterInterface $dateFormatter;

  /**
   * {@inheritdoc}
   */
  public static function createInstance(ContainerInterface $container, $entity_type): NexiXpayTransactionListBuilder|EntityListBuilder|EntityHandlerInterface {
    /** @var static $instance */
    $instance = parent::createInstance($container, $entity_type);

    /** @var \Drupal\Core\Datetime\DateFormatterInterface $date_formatter */
    $date_formatter = $container->get('date.formatter');
    $instance->dateFormatter = $date_formatter;

    return $instance;
  }

  /**
   * {@inheritdoc}
   *
   * @return array<string, mixed>
   *   A render array structure of header strings.
   */
  public function buildHeader(): array {
    $header['id'] = $this->t('ID');
    $header['status'] = $this->t('Status');
    $header['token_expires'] = $this->t('Expire date');
    $header['mode'] = $this->t('Mode');
    $header['amount'] = $this->t('Amount');
    $header['merchant_reference'] = $this->t('Merchant ref');
    $header['changed'] = $this->t('Updated');
    $header['operations'] = $this->t('Operations');
    return $header + parent::buildHeader();
  }

  /**
   * {@inheritdoc}
   *
   * @return array<string, mixed>
   *   A render array structure of fields for this entity.
   */
  public function buildRow(EntityInterface $entity): array {
    /** @var \Drupal\nexi_xpay\Entity\NexiXpayTransaction $entity */
    $row['id'] = Link::createFromRoute(
      (string) $entity->id(),
      'entity.nexi_xpay_transaction.edit_form',
      ['nexi_xpay_transaction' => $entity->id()]
    );
    $row['status'] = $entity->getStatus();
    $row['token_expires'] = $entity->getStatus() === NexiXpayTransactionInterface::STATUS_PENDING
        ? $this->dateFormatter->format((int) $entity->getTokenExpiresTime(), 'short')
        : '-';
    $row['mode'] = $entity->getMode();
    $row['amount'] = $entity->getAmountFormatted() . ' ' . $entity->getCurrency();
    $row['merchant_reference'] = $entity->getMerchantReference();
    $row['changed'] = $this->dateFormatter->format((int) $entity->getChangedTime(), 'short');
    $row['operations']['data'] = $this->buildOperations($entity);
    return $row + parent::buildRow($entity);
  }

  /**
   * {@inheritdoc}
   *
   * @return array<string, array{title: string, url: \Drupal\Core\Url, weight: int, attributes?: array<string, mixed>}>
   *   An associative array of operation link data for this list, keyed by
   *   operation name, containing the following key-value pairs:
   *   - title: The localized title of the operation.
   *   - url: An instance of \Drupal\Core\Url for the operation URL.
   *   - weight: The weight of this operation.
   */
  public function getDefaultOperations(EntityInterface $entity): array {
    $operations = parent::getDefaultOperations($entity);

    /** @var \Drupal\nexi_xpay\Entity\NexiXpayTransactionInterface $entity */
    $status = $entity->getStatus();
    $token = $entity->getPublicToken();

    // Link "Pay" only if pending and token present.
    if ($status === NexiXpayTransactionInterface::STATUS_PENDING && $token !== '') {
      $operations['pay'] = [
        'title' => $this->t('Pay'),
        'url' => Url::fromRoute('nexi_xpay.pay', [
          'nexi_xpay_transaction' => $entity->id(),
        ], [
          'query' => ['t' => $token],
          'absolute' => TRUE,
        ]),
        'weight' => -20,
        'attributes' => [
          'target' => '_blank',
          'rel' => 'noopener noreferrer',
        ],
      ];
    }

    // Link "Regenerate token" only if pending.
    if ($status === NexiXpayTransactionInterface::STATUS_PENDING) {
      $operations['regenerate_token'] = [
        'title' => $this->t('Regenerate token'),
        'url' => Url::fromRoute('nexi_xpay.regenerate_token', [
          'nexi_xpay_transaction' => $entity->id(),
        ]),
        'weight' => 5,
      ];
    }

    return $operations;
  }

}
