<?php

declare(strict_types=1);

namespace Drupal\nexi_xpay\Plugin\XpayMode;

use Drupal\Core\Routing\TrustedRedirectResponse;
use Drupal\Core\Url;
use Drupal\nexi_xpay\Entity\NexiXpayTransactionInterface;
use Drupal\nexi_xpay\Plugin\XpayMode\Attribute\XpayMode;
use Drupal\nexi_xpay\Service\NexiXpayClientInterface;
use Drupal\nexi_xpay\Value\ModeHandleResult;
use Drupal\nexi_xpay\Value\ModeStartResult;
use Symfony\Component\HttpFoundation\Request;

/**
 * Hosted Payment Page (HPP) mode implementation.
 *
 * This mode integrates Nexi XPay using the Hosted Payment Page flow and is
 * designed to work out-of-the-box for site builders, while remaining
 * extensible for developers via the NexiXpay mode plugin system.
 *
 * How the result objects are used in the payment flow:
 *
 * - Start flow (startPayment):
 *   1) The controller calls
 *      NexiXpayManager::startPayment($transaction, $request).
 *   2) The manager delegates to this mode plugin.
 *   3) This mode returns a ModeStartResult:
 *      - If ModeStartResult::$response is set, the controller returns it
 *        immediately (typically a redirect to Nexi / HPP).
 *      - Otherwise, the controller renders ModeStartResult::$renderArray as an
 *        intermediate Drupal page.
 *
 * - Return/Notify flow (handleReturn / handleNotify):
 *   1) The controller calls NexiXpayManager::handleReturn() or handleNotify().
 *   2) The manager delegates to this mode plugin.
 *   3) This mode returns a ModeHandleResult containing:
 *      - $newStatus (optional) computed from the callback/payload, and
 *      - $context (extra metadata for logging and UI messages).
 *    4) The controller applies the state transition using
 *       TransactionStatusUpdater (idempotent and monotonic), then:
 *       - builds a human-friendly return page for the customer, or
 *       - returns a constant 204 response for server-to-server notifications.
 *
 * This separation keeps the controller thin, concentrates gateway-specific
 * logic inside the mode, and ensures consistent transaction updates across
 * all modes.
 *
 * @phpstan-type NexiXpayOperation array{
 *   orderId?: string,
 *   operationResult?: string,
 *   operationId?: string,
 *   channel?: string,
 *   operationType?: string,
 *   operationTime?: string,
 *   paymentCircuit?: string,
 *   paymentInstrumentInfo?: string,
 * }
 *
 * @phpstan-type NexiXpayNotification array{
 *   eventId?: string,
 *   eventTime?: string,
 *   securityToken?: string,
 *   operation?: NexiXpayOperation,
 * }
 */
#[XpayMode(
  id: "hpp_redirect",
  label: "HPP Redirect",
  description: "Hosted Payment Page / browser redirect (server-side)."
)]
final class HostedPaymentPageMode extends NexiXpayModeBase {

  /**
   * The plugin manager injects the client interface.
   */
  private NexiXpayClientInterface $client;

  /**
   * Constructs a \Drupal\nexi_xpay\Plugin\NexiXpayMode objet.
   *
   * @param array<string, mixed> $configuration
   *   An array of configuration key-value pairs.
   * @param string $plugin_id
   *   The plugin_id for the plugin instance.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   */
  public function __construct(array $configuration, string $plugin_id, mixed $plugin_definition) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);

    if (!isset($configuration['client']) || !$configuration['client'] instanceof NexiXpayClientInterface) {
      throw new \InvalidArgumentException('Missing NexiXpayClientInterface in plugin configuration.');
    }

    $this->client = $configuration['client'];
  }

  /**
   * {@inheritdoc}
   */
  public function buildPayPage(NexiXpayTransactionInterface $transaction, Request $request): array {
    $token = (string) $request->query->get('t', '');

    $start_url = Url::fromRoute('nexi_xpay.start', [
      'nexi_xpay_transaction' => $transaction->id(),
    ], [
      'absolute' => TRUE,
      'query' => ['t' => $token],
    ])->toString();

    return [
      '#theme' => 'nexi_xpay_hpp_pay',
      '#tx' => [
        'id' => (int) $transaction->id(),
        'status' => $transaction->getStatus(),
        'amount' => $transaction->getAmountFormatted(),
        'currency' => $transaction->getCurrency(),
        'merchant_reference' => $transaction->getMerchantReference(),
      ],
      '#start_url' => $start_url,
      '#cache' => ['max-age' => 0],
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function buildReturnPage(NexiXpayTransactionInterface $transaction, Request $request, ModeHandleResult $handleResult): array {
    return [
      '#theme' => 'nexi_xpay_hpp_return',
      '#tx' => [
        'id' => (int) $transaction->id(),
        'status' => $transaction->getStatus(),
      ],
      '#message' => $handleResult->message ?: 'Return handled.',
      '#cache' => ['max-age' => 0],
    ];
  }

  /**
   * {@inheritdoc}
   *
   * @throws \Drupal\Core\Entity\EntityStorageException|\Random\RandomException
   *   Catches all exceptions and returns a generic error message.
   */
  public function startPayment(NexiXpayTransactionInterface $transaction, Request $request): ModeStartResult {
    // Security token is passed via querystring.
    $token = (string) $request->query->get('t', '');

    // Payment data.
    $transactionId = $this->loggerContext['transaction'] = (int) $transaction->id();
    $orderId = $this->loggerContext['orderId'] = $transaction->getOrderId();
    $amount = $this->loggerContext['amount'] = $transaction->getAmount();
    $currency = $this->loggerContext['currency'] = $transaction->getCurrency();

    // Browser return/cancel MUST be public and MUST NOT
    // contain token/querystring.
    $returnUrl = Url::fromRoute('nexi_xpay.return', [
      'orderId' => $orderId,
    ], ['absolute' => TRUE])->toString();

    $cancelUrl = Url::fromRoute('nexi_xpay.cancel', [
      'orderId' => $orderId,
    ], ['absolute' => TRUE])->toString();

    // Server-to-server callback with token in the URL path.
    $notifyToken = $transaction->getNotifyToken();
    $notifyUrl = Url::fromRoute('nexi_xpay.notify', [
      'nexi_xpay_transaction' => $transactionId,
      'token' => $notifyToken,
    ], ['absolute' => TRUE])->toString();

    // Basic validation for Nexi constraints.
    if ($orderId === '' || !preg_match('/^[a-z0-9]{1,18}$/', $orderId)) {
      $this->logger->warning('HPP start rejected: invalid orderId.', $this->loggerContext);
      return ModeStartResult::render([
        '#markup' => 'Unable to start payment: invalid orderId.',
        '#cache' => ['max-age' => 0],
      ]);
    }

    if ($amount <= 0) {
      $this->logger->warning('HPP start rejected: invalid amount.', $this->loggerContext);
      return ModeStartResult::render([
        '#markup' => 'Unable to start payment: invalid amount.',
        '#cache' => ['max-age' => 0],
      ]);
    }

    if ($token === '') {
      // Should not happen if accessed via pay link, but keep it explicit.
      $this->logger->warning('HPP start rejected: missing public token.', $this->loggerContext);
      return ModeStartResult::render([
        '#markup' => 'Missing token.',
        '#cache' => ['max-age' => 0],
      ]);
    }

    // Build payload as per Nexi POST /orders/hpp.
    // Minimal required fields:
    // - order: { orderId, amount, currency }
    // - paymentSession: { actionType, amount, language }
    // - resultUrl, cancelUrl
    // - notificationUrl (recommended for your flow)
    $payload = [
      'order' => [
        'orderId' => $orderId,
        'amount' => (string) $amount,
        'currency' => $currency,
        'description' => 'Nexi XPay transaction ' . $transactionId,
      ],
      'paymentSession' => [
        'actionType' => 'PAY',
        'amount' => (string) $amount,
        // ISO 639-2; default in docs is "ita".
        // @todo determine language from user session.
        'language' => 'ita',
        'resultUrl' => $returnUrl,
        'cancelUrl' => $cancelUrl,
        'notificationUrl' => $notifyUrl,
      ],
    ];

    $this->logger->debug(
      'HPP /orders/hpp request payload.',
      array_merge($this->loggerContext, ['payload' => $payload])
    );

    try {
      // Call Nexi (base_url already points to .../api/v1,
      // so path is /orders/hpp).
      $res = $this->client->requestJson('POST', '/orders/hpp', [
        'json' => $payload,
        'timeout' => 20,
        'connect_timeout' => 10,
      ]);

      $status = $this->loggerContext['httpStatus'] = (int) $res['status'];
      $this->loggerContext['endpoint'] = '/orders/hpp';

      $this->logger->debug(
        'HPP /orders/hpp response body.',
        array_merge($this->loggerContext, ['body' => (string) $res['body']])
      );

      /** @var array<string, string> $data */
      $data = $res['json'];

      if ($status !== 200) {
        $this->logger->error('HPP /orders/hpp returned unexpected HTTP status.', $this->loggerContext);
        return ModeStartResult::render([
          '#markup' => 'Unable to start payment (gateway error).',
          '#cache' => ['max-age' => 0],
        ]);
      }

      $hostedPage = ($data['hostedPage'] ?? '');
      $securityToken = ($data['securityToken'] ?? '');

      if ($hostedPage === '' || $securityToken === '') {
        $this->logger->error('HPP /orders/hpp response missing hostedPage/securityToken.', $this->loggerContext);
        return ModeStartResult::render([
          '#markup' => 'Unable to start payment (invalid gateway response).',
          '#cache' => ['max-age' => 0],
        ]);
      }

      // Persist correlation fields.
      $transaction->set('order_id', $orderId);
      $transaction->set('security_token', $securityToken);
      if ($transaction->getStatus() === NexiXpayTransactionInterface::STATUS_PENDING) {
        $transaction->setStatus(NexiXpayTransactionInterface::STATUS_PROCESSING);
      }
      $transaction->save();

      $this->logger->info('HPP start successful; redirecting to hosted page.', $this->loggerContext);

      return ModeStartResult::response(new TrustedRedirectResponse($hostedPage));
    }
    catch (\Throwable $e) {
      $this->logger->error(
        'HPP /orders/hpp failed.',
        array_merge(
          $this->loggerContext,
          ['endpoint' => '/orders/hpp', 'error' => $e->getMessage()]
        )
      );

      return ModeStartResult::render([
        '#markup' => 'Unable to start payment (exception).',
        '#cache' => ['max-age' => 0],
      ]);
    }
  }

  /**
   * {@inheritdoc}
   *
   * @throws \Drupal\Core\Entity\EntityStorageException
   */
  public function handleReturn(NexiXpayTransactionInterface $transaction, Request $request): ModeHandleResult {
    $this->loggerContext['transaction'] = (int) $transaction->id();
    $currentStatus = $this->loggerContext['currentStatus'] = $transaction->getStatus();
    $orderId = $this->loggerContext['orderId'] = $transaction->getOrderId();

    // Final states: do nothing (idempotent and safe).
    if ($transaction->isInFinalStatus()) {
      $this->logger->info('Return received but transaction is already in a final state.', $this->loggerContext);

      return ModeHandleResult::noop('Return received (already final).', [
        'source' => 'return',
        'currentStatus' => $currentStatus,
        'query' => $request->query->all(),
      ]);
    }

    $this->logger->info('Return received.', $this->loggerContext);

    // @todo move to a dedicated class
    // 2) Try to retrieve the authoritative outcome via GET /orders/{orderId}.
    try {
      $resp = $this->client->getOrder($orderId);

      $status = $this->loggerContext['httpStatus'] = (int) $resp['status'];
      $endpoint = $this->loggerContext['endpoint'] = '/orders/' . $orderId;
      $jsonData = (array) $resp['json'];
      $outcome = $this->loggerContext['outcome'] = $this->extractOutcomeFromGetOrder($jsonData);
      $mapped = $this->loggerContext['newStatus'] = $this->mapOutcomeToInternalStatus($outcome);

      // If I'm not able to map, I stay in processing (if pending).
      if ($mapped === NULL) {
        $this->logger->warning('Return handled via GET /orders but outcome is not mapped.', $this->loggerContext);

        if ($currentStatus === NexiXpayTransactionInterface::STATUS_PENDING) {
          return ModeHandleResult::status(
            NexiXpayTransactionInterface::STATUS_PROCESSING,
            'Return received. Awaiting confirmation from the payment gateway...',
            [
              'source' => 'return_get_order',
              'orderId' => $orderId,
              'outcome' => $outcome,
              'query' => $request->query->all(),
            ]
          );
        }

        return ModeHandleResult::noop(
          'Return received. Order not final yet.',
          [
            'source' => 'return_get_order',
            'orderId' => $orderId,
            'outcome' => $outcome,
            'currentStatus' => $currentStatus,
            'query' => $request->query->all(),
          ]
        );
      }

      $this->logger->info('Return processed via GET /orders.', $this->loggerContext);

      $this->logger->debug(
        'GET /orders/{orderId} response data.',
        array_merge(
          $this->loggerContext,
          ['body' => (string) $resp['body'], 'payload' => $jsonData]
        )
      );

      // Apply transition (the state machine and idempotency
      // is handled by the manager/updater).
      return ModeHandleResult::status(
        $mapped,
        'Return processed via GET /orders.',
        [
          'source' => 'return_get_order',
          'orderId' => $orderId,
          'outcome' => $outcome,
          'query' => $request->query->all(),
        ]
      );
    }
    catch (\Throwable $e) {
      $this->logger->error(
        'Return received but GET /orders failed.',
        array_merge($this->loggerContext, ['error' => $e->getMessage()])
      );

      if ($currentStatus === NexiXpayTransactionInterface::STATUS_PENDING) {
        return ModeHandleResult::status(
          NexiXpayTransactionInterface::STATUS_PROCESSING,
          'Return received. Awaiting confirmation from the payment gateway...',
          [
            'source' => 'return',
            'orderId' => $orderId,
            'error' => $e->getMessage(),
            'query' => $request->query->all(),
          ]
        );
      }

      return ModeHandleResult::noop('Return received (GET /orders failed).', [
        'source' => 'return',
        'orderId' => $orderId,
        'error' => $e->getMessage(),
        'currentStatus' => $currentStatus,
        'query' => $request->query->all(),
      ]);
    }
  }

  /**
   * Extract a meaningful outcome from GET /orders/{orderId}.
   *
   * Primary source (per docs): orderStatus.operations[] and
   * its operationResult. Fallbacks are kept minimal to avoid confusing
   * "operation type" with "result".
   *
   * @param array<string, mixed> $data
   *   Decoded JSON response.
   *
   * @return string
   *   Uppercase outcome/status candidate, or empty string.
   */
  private function extractOutcomeFromGetOrder(array $data): string {
    // 1) Preferred location: orderStatus.operations[] (documented structure).
    $operations = $this->getOperationsArray($data['orderStatus'] ?? NULL);

    // 2) Fallback: some payloads may expose operations at top-level.
    if ($operations === NULL) {
      $operations = $this->getOperationsArray($data);
    }

    // If we have operations, pick the most recent one and
    // extract operationResult.
    if (is_array($operations) && $operations !== []) {
      $last = $this->pickMostRecentOperation($operations);

      if (is_array($last)) {
        $result = $this->normalizeOutcome($last['operationResult'] ?? NULL);
        if ($result !== '') {
          return $result;
        }

        // Minimal fallback: sometimes a nested additionalData.status
        // is present.
        if (isset($last['additionalData']) && is_array($last['additionalData'])) {
          $result = $this->normalizeOutcome($last['additionalData']['status'] ?? NULL);
          if ($result !== '') {
            return $result;
          }
        }
      }
    }

    // 3) Last-resort fallbacks (do NOT use operationType here).
    if (isset($data['orderStatus']) && is_array($data['orderStatus'])) {
      $result = $this->normalizeOutcome($data['orderStatus']['status'] ?? NULL);
      if ($result !== '') {
        return $result;
      }
    }

    if (isset($data['order']) && is_array($data['order'])) {
      $result = $this->normalizeOutcome($data['order']['status'] ?? NULL);
      if ($result !== '') {
        return $result;
      }
    }

    return '';
  }

  /**
   * Return operations[] if present, otherwise NULL.
   *
   * @param mixed $container
   *   Typically, $data['orderStatus'] or $data (top-level).
   *
   * @return array<int, mixed>|null
   *   The operations array if found.
   */
  private function getOperationsArray(mixed $container): ?array {
    if (!is_array($container)) {
      return NULL;
    }
    $ops = $container['operations'] ?? NULL;
    if (!is_array($ops) || !array_is_list($ops)) {
      return NULL;
    }

    return $ops;
  }

  /**
   * Pick the most recent operation.
   *
   * If operationTime is available, use it; otherwise fall back
   * to the last element.
   *
   * @param array<int, mixed> $operations
   *   Operations array.
   *
   * @return mixed
   *   The selected operation entry.
   */
  private function pickMostRecentOperation(array $operations): mixed {
    $best = NULL;
    $bestTs = NULL;

    foreach ($operations as $op) {
      if (!is_array($op)) {
        continue;
      }
      $t = $op['operationTime'] ?? NULL;
      if (!is_string($t) || $t === '') {
        continue;
      }
      $ts = strtotime($t);
      if ($ts === FALSE) {
        continue;
      }
      if ($bestTs === NULL || $ts >= $bestTs) {
        $bestTs = $ts;
        $best = $op;
      }
    }

    if ($best !== NULL) {
      return $best;
    }

    // Fallback: keep existing behavior if no timestamps are usable.
    return end($operations);
  }

  /**
   * Normalize an outcome value to an uppercase string.
   */
  private function normalizeOutcome(mixed $value): string {
    if (!is_scalar($value)) {
      return '';
    }
    return strtoupper(trim((string) $value));
  }

  /**
   * Map Nexi outcome/order status into internal transaction status.
   *
   * @param string $outcome
   *   Nexi outcome status.
   *
   * @return string|null
   *   Internal status, or NULL if not mapped.
   */
  private function mapOutcomeToInternalStatus(string $outcome): ?string {
    $o = strtoupper(trim($outcome));
    if ($o === '') {
      return NULL;
    }

    // Final: paid.
    if (in_array($o, ['EXECUTED', 'CAPTURED', 'SUCCESS', 'OK'], TRUE)) {
      return NexiXpayTransactionInterface::STATUS_PAID;
    }

    // Not final (processing)
    if (in_array($o, ['AUTHORIZED', 'PENDING', 'PROCESSING', 'IN_PROGRESS', 'CREATED', 'THREEDS_VALIDATED'], TRUE)) {
      return NexiXpayTransactionInterface::STATUS_PROCESSING;
    }

    // Final: failed.
    if (in_array($o, ['DECLINED', 'DENIED_BY_RISK', 'FAILED', 'KO', 'THREEDS_FAILED', 'ERROR'], TRUE)) {
      return NexiXpayTransactionInterface::STATUS_FAILED;
    }

    // Final: cancelled.
    if (in_array($o, ['CANCELED', 'CANCELLED', 'VOIDED', 'ABORTED'], TRUE)) {
      return NexiXpayTransactionInterface::STATUS_CANCELLED;
    }

    return NULL;
  }

  /**
   * {@inheritdoc}
   *
   * @throws \Drupal\Core\Entity\EntityStorageException
   */
  public function handleNotify(NexiXpayTransactionInterface $transaction, Request $request): ModeHandleResult {
    $this->loggerContext['transaction'] = (int) $transaction->id();
    $currentStatus = $this->loggerContext['currentStatus'] = $transaction->getStatus();
    $currentOrderId = $this->loggerContext['orderId'] = $transaction->getOrderId();

    // Parse payload: prefer JSON, fallback to form params.
    $raw = (string) $request->getContent();
    $data = [];
    $content_type = (string) $request->headers->get('content-type', '');

    if (str_contains($content_type, 'application/json')) {
      $decoded = json_decode($raw, TRUE);
      if (is_array($decoded)) {
        /** @var NexiXpayNotification $data */
        $data = $decoded;
      }
    }

    if (!$data) {
      // Fallback: x-www-form-urlencoded or multipart.
      $fallback = $request->request->all();
      if (!empty($fallback)) {
        /** @var NexiXpayNotification $data */
        $data = $fallback;
      }
      else {
        $this->logger->warning('Notify ignored: no payload found.', $this->loggerContext);

        return ModeHandleResult::noop('Notify ignored: no payload found.', [
          'eventId' => 'unknown',
          'orderId' => 'unknown',
          'operationResult' => 'unknown',
        ]);
      }
    }

    // Normalize raw: if empty but data exists, encode it for debugging.
    if ($raw === '' && $data !== []) {
      $encoded = $this->logger->jsonSafeEncode($data);
      if ($encoded !== '') {
        $raw = $encoded;
      }
    }

    // Extract standard fields.
    $securityToken = is_string($data['securityToken'] ?? NULL) ? $data['securityToken'] : '';
    $eventId = $this->loggerContext['eventId'] =
      is_string($data['eventId'] ?? NULL) ? $data['eventId'] : '';

    $operation = (isset($data['operation'])) ? $data['operation'] : [];
    $orderIdFromNotify = is_string($operation['orderId'] ?? NULL) ? $operation['orderId'] : '';
    $operationResult = $this->loggerContext['operationResult'] =
      strtoupper(is_string($operation['operationResult'] ?? NULL) ? $operation['operationResult'] : '');

    $this->logger->info('Notify received.', $this->loggerContext);
    $this->logger->debug(
      'Notify decoded payload.',
      array_merge($this->loggerContext, ['payload' => $raw])
    );

    // Security token validation (if configured).
    $expected = $transaction->getSecurityToken();
    if ($expected !== '' && $securityToken !== '' && !hash_equals($expected, $securityToken)) {
      $this->logger->warning('Notify ignored: security token mismatch.', $this->loggerContext);
      return ModeHandleResult::noop('Notify ignored: security token mismatch.', [
        'eventId' => $eventId,
        'orderId' => $orderIdFromNotify,
        'operationResult' => $operationResult,
      ]);
    }

    // Strict mode: security_token must be set during start/init (Step 3).
    if ($expected === '') {
      $this->logger->warning('Notify ignored: security token not configured on transaction.', $this->loggerContext);
      return ModeHandleResult::noop('Notify ignored: security token not configured.', [
        'eventId' => $eventId,
        'orderId' => $orderIdFromNotify,
        'operationResult' => $operationResult,
      ]);
    }

    // 5) Map operationResult -> internal status.
    $newStatus = $this->loggerContext['newStatus'] = $this->mapOperationResultToStatus($operationResult);

    // Should mapping prove impossible,
    // just provide an idempotent acknowledgment.
    if ($newStatus === NULL) {
      $this->logger->warning('Notify received but operationResult is not mapped.', $this->loggerContext);
      return ModeHandleResult::noop('Notify received (unmapped).', [
        'eventId' => $eventId,
        'orderId' => $orderIdFromNotify,
        'operationResult' => $operationResult,
      ]);
    }

    // 6) Idempotent + monotonic transitions.
    if (!$this->isTransitionAllowed($currentStatus, $newStatus)) {
      $this->logger->warning('Notify ignored by state machine.', $this->loggerContext);

      return ModeHandleResult::noop('Notify received (ignored by state machine).', [
        'eventId' => $eventId,
        'orderId' => $orderIdFromNotify,
        'operationResult' => $operationResult,
        'currentStatus' => $currentStatus,
        'newStatus' => $newStatus,
      ]);
    }

    $this->logger->info('Notify processed.', $this->loggerContext);

    return ModeHandleResult::status($newStatus, 'Notify processed.', [
      'eventId' => $eventId,
      'orderId' => $orderIdFromNotify,
      'operationResult' => $operationResult,
    ]);
  }

  /**
   * Conservative mapping from Nexi operationResult to internal status.
   *
   * @param string $operationResult
   *   Match Nexi result codes with internal statuses.
   *
   * @return string|null
   *   Internal status, or NULL if not applicable.
   */
  private function mapOperationResultToStatus(string $operationResult): ?string {
    return match ($operationResult) {
      'EXECUTED' => NexiXpayTransactionInterface::STATUS_PAID,

      // Authorization/3DS steps: keep it non-final.
      'AUTHORIZED', 'PENDING', 'THREEDS_VALIDATED' => NexiXpayTransactionInterface::STATUS_PROCESSING,

      // Negative outcomes.
      'DECLINED', 'DENIED_BY_RISK', 'THREEDS_FAILED', 'FAILED' => NexiXpayTransactionInterface::STATUS_FAILED,

      // Customer/system cancellation-like outcomes.
      'CANCELED', 'VOIDED' => NexiXpayTransactionInterface::STATUS_CANCELLED,

      default => NULL,
    };
  }

  /**
   * Prevent non-monotonic transitions (idempotent and safe).
   *
   * @param string $current
   *   Current state.
   * @param string $next
   *   Desired state.
   *
   * @return bool
   *   TRUE if allowed, FALSE otherwise.
   */
  private function isTransitionAllowed(string $current, string $next): bool {
    // The same state is always idempotent OK (updater will not-op anyway).
    if ($current === $next) {
      return TRUE;
    }

    // Final states: do not change them via notification.
    $final = [
      NexiXpayTransactionInterface::STATUS_PAID,
      NexiXpayTransactionInterface::STATUS_FAILED,
      NexiXpayTransactionInterface::STATUS_CANCELLED,
      NexiXpayTransactionInterface::STATUS_EXPIRED,
    ];
    if (in_array($current, $final, TRUE)) {
      return FALSE;
    }

    // From pending/processing you can go to anything mapped.
    return in_array($current, [
      NexiXpayTransactionInterface::STATUS_PENDING,
      NexiXpayTransactionInterface::STATUS_PROCESSING,
    ], TRUE);
  }

}
