<?php

declare(strict_types=1);

namespace Drupal\nexi_xpay\Plugin\XpayMode;

use Drupal\Component\Plugin\PluginBase;
use Drupal\nexi_xpay\Entity\NexiXpayTransactionInterface;
use Drupal\nexi_xpay\Value\ModeHandleResult;
use Drupal\nexi_xpay\Value\ModeStartResult;
use Psr\Log\LoggerInterface;
use Symfony\Component\HttpFoundation\Request;

/**
 * Base class for NexiXpayMode plugins.
 */
abstract class NexiXpayModeBase extends PluginBase implements NexiXpayModeInterface {

  /**
   * The logger.
   *
   * @var \Drupal\nexi_xpay\Service\Logger
   */
  protected LoggerInterface $logger;

  /**
   * The logger context.
   *
   * @var array{amount?: int, body?: string, correlationId?: string, currentStatus?: string, currency?: string, endpoint?: string, error?: string, eventId?: string, httpStatus?: int, method?: string, mode?: string, newStatus?: string|null, payload?: string, operationResult?: string, orderId?: string, outcome?: string, transaction?: int, url?: string}
   */
  protected array $loggerContext = [];

  /**
   * {@inheritdoc}
   *
   * @param array<string, mixed> $configuration
   *   Plugin configuration.
   * @param string $plugin_id
   *   Plugin ID.
   * @param mixed $plugin_definition
   *   Plugin definition.
   */
  public function __construct(array $configuration, string $plugin_id, mixed $plugin_definition) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);

    if (!isset($configuration['logger']) || !$configuration['logger'] instanceof LoggerInterface) {
      throw new \InvalidArgumentException('Missing LoggerInterface in plugin configuration.');
    }
    /** @var \Drupal\nexi_xpay\Service\Logger $logger */
    $logger = $configuration['logger'];
    $this->logger = $logger;

    $this->loggerContext = $this->logger->setupContext();
    $this->loggerContext['mode'] = $this->getPluginId();
  }

  /**
   * {@inheritDoc}
   */
  public function startPayment(NexiXpayTransactionInterface $transaction, Request $request): ModeStartResult {
    return ModeStartResult::render([
      '#markup' => 'Start payment not implemented for this mode.',
      '#cache' => ['max-age' => 0],
    ]);
  }

  /**
   * Base implementation.
   */
  public function handleReturn(NexiXpayTransactionInterface $transaction, Request $request): ModeHandleResult {
    return ModeHandleResult::noop('Return not implemented.');
  }

  /**
   * {@inheritDoc}
   */
  public function handleNotify(NexiXpayTransactionInterface $transaction, Request $request): ModeHandleResult {
    return ModeHandleResult::noop('Notify not implemented.');
  }

  /**
   * {@inheritDoc}
   */
  public function buildPayPage(NexiXpayTransactionInterface $transaction, Request $request): array {
    return [
      '#markup' => 'Pay page not implemented for this mode.',
    ];
  }

  /**
   * {@inheritDoc}
   */
  public function buildReturnPage(NexiXpayTransactionInterface $transaction, Request $request, ModeHandleResult $handleResult): array {
    return [
      '#markup' => $handleResult->message ?: 'Return handled.',
    ];
  }

}
