<?php

declare(strict_types=1);

namespace Drupal\nexi_xpay\Service;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Logger\RfcLoggerTrait;
use Psr\Log\LoggerInterface;

/**
 * Provides a decorator for the Drupal logger specific to Nexi XPay.
 *
 * This class intercepts log messages to format them uniformly,
 * automatically integrating transaction metadata into the message
 * and handling placeholders for the Drupal logging system.
 */
class Logger implements LoggerInterface {
  use RfcLoggerTrait;

  /**
   * Max length for logged payload/body strings.
   */
  public const int MAX_LEN = 60000;

  /**
   * If true, log payloads in the log messages.
   */
  private bool $logPayloads;

  /**
   * Context keys that are allowed to be logged.
   */
  private const array ALLOWED_LOG_FIELDS_KEYS = [
    'amount',
    'body',
    'correlationId',
    'currentStatus',
    'currency',
    'endpoint',
    'error',
    'eventId',
    'httpStatus',
    'method',
    'mode',
    'newStatus',
    'payload',
    'operationResult',
    'orderId',
    'outcome',
    'transaction',
    'url',
  ];

  /**
   * The logger to decorate constructor.
   *
   * @param \Psr\Log\LoggerInterface $logger
   *   The logger interface.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $configFactory
   *   The config factory interface.
   */
  public function __construct(
    private readonly LoggerInterface $logger,
    private readonly ConfigFactoryInterface $configFactory,
  ) {
    $this->logPayloads = (bool) $this->configFactory->get('nexi_xpay.settings')->get('logging.log_payloads');
  }

  /**
   * Returns the default context set for all messages.
   *
   * @return array{amount?: int, body?: string, correlationId?: string, currentStatus?: string, currency?: string, endpoint?: string, error?: string, eventId?: string, httpStatus?: int, method?: string, mode?: string, newStatus?: string|null, payload?: string, operationResult?: string, orderId?: string, outcome?: string, transaction?: int, url?: string}
   *   Default context set for all messages.
   */
  public function setupContext(): array {
    return [
      'amount' => 0,
      'body' => '',
      'correlationId' => '',
      'currentStatus' => '',
      'currency' => '',
      'endpoint' => '',
      'error' => '',
      'eventId' => '',
      'httpStatus' => 0,
      'method' => '',
      'mode' => '',
      'newStatus' => '',
      'payload' => '',
      'operationResult' => '',
      'orderId' => '',
      'outcome' => '',
      'transaction' => 0,
      'url' => '',
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function log($level, string|\Stringable $message, array $context = []): void {
    if (!$this->logPayloads && isset($context['payload'])) {
      unset($context['payload']);
    }
    if (!$this->logPayloads && isset($context['body'])) {
      unset($context['body']);
    }

    $formatted = $this->formatForDrupalLog($message, $context);

    $this->logger->log(
      $level,
      $formatted['message'],
      $formatted['psr3'] + $formatted['drupal']
    );
  }

  /**
   * {@inheritdoc}
   */
  public function debug(\Stringable|string $message, array $context = []): void {
    if ($this->logPayloads && isset($context['payload'])) {
      $raw_payload = $context['payload'];
      $raw_payload = $this->jsonSafeEncode($raw_payload);
      $raw_payload = $this->truncateString($raw_payload);
      $context['payload'] = $raw_payload;
    }
    else {
      unset($context['payload']);
    }

    if ($this->logPayloads && isset($context['body'])) {
      $raw_body = (string) $context['body'];
      $raw_body = $this->truncateString($raw_body);
      $context['body'] = $raw_body;
    }
    else {
      unset($context['body']);
    }

    $this->log('debug', $message, $context);
  }

  /**
   * Build a DBLog-friendly message with placeholders AND a PSR-3 context.
   *
   * @param string|\Stringable $baseMessage
   *   The base message.
   * @param array<string, scalar> $context
   *   Canonical context keys: transaction, mode, orderId, endpoint, httpStatus,
   *   eventId, operationResult, currentStatus, newStatus, error.
   *
   * @return array{message: string, drupal: array<string, scalar>, psr3: array<string, scalar>}
   *   The message and context.
   */
  public function formatForDrupalLog(string|\Stringable $baseMessage, array $context): array {
    $fields = self::ALLOWED_LOG_FIELDS_KEYS;

    $parts = [$baseMessage];
    $drupal = [];
    $psr3 = $context;

    foreach ($fields as $fieldKey) {
      $fieldData = $context[$fieldKey] ?? NULL;
      if (empty($fieldData)) {
        continue;
      }

      $placeholder = '@' . $fieldKey;
      $parts[] = '<br><strong>' . $fieldKey . '</strong>: ' . $placeholder;

      $drupal[$placeholder] = (string) $fieldData;
    }

    return [
      'message' => implode(' ', $parts),
      'drupal' => $drupal,
      'psr3' => $psr3,
    ];
  }

  /**
   * Truncate a string to a maximum length.
   *
   * @param string $str
   *   The string to truncate.
   * @param int $max_len
   *   The maximum length.
   *
   * @return string
   *   The truncated string.
   */
  public function truncateString(string $str, int $max_len = self::MAX_LEN): string {
    if ($str === '') {
      return '';
    }
    return mb_substr($str, 0, $max_len);
  }

  /**
   * Encode payload safely.
   *
   * @param mixed $payload
   *   The payload to encode.
   *
   * @return string
   *   The encoded payload.
   */
  public function jsonSafeEncode(mixed $payload): string {
    $json = json_encode($payload, JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE);
    return is_string($json) ? $json : '';
  }

}
