<?php

declare(strict_types=1);

namespace Drupal\nexi_xpay\Service;

use Drupal\Component\Uuid\UuidInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use GuzzleHttp\ClientInterface;
use Psr\Log\LoggerInterface;

/**
 * Nexi Xpay client.
 */
final class NexiXpayClient implements NexiXpayClientInterface {

  /**
   * The logger context.
   *
   * @var array{amount?: int, body?: string, correlationId?: string, currentStatus?: string, currency?: string, endpoint?: string, error?: string, eventId?: string, httpStatus?: int, method?: string, mode?: string, newStatus?: string|null, payload?: string, operationResult?: string, orderId?: string, outcome?: string, transaction?: int, url?: string}
   */
  protected array $loggerContext;

  /**
   * The constructor.
   *
   * @param \GuzzleHttp\ClientInterface $httpClient
   *   HTTP client.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $configFactory
   *   Config factory.
   * @param \Drupal\nexi_xpay\Service\Logger $logger
   *   Logger.
   * @param \Drupal\Component\Uuid\UuidInterface $uuid
   *   UUID generator.
   */
  public function __construct(
    private readonly ClientInterface $httpClient,
    private readonly ConfigFactoryInterface $configFactory,
    private readonly LoggerInterface $logger,
    private readonly UuidInterface $uuid,
  ) {
    $this->loggerContext = $this->logger->setupContext();
    $this->loggerContext['mode'] = 'NexiXpayClient';
  }

  /**
   * {@inheritDoc}
   */
  public function getEnvironment(): string {
    $config = $this->configFactory->get('nexi_xpay.settings');

    /** @var string $env */
    $env = $config->get('environment');

    return $env;
  }

  /**
   * Returns the active configuration.
   *
   * @return array{base_url: string, api_key: string, merchant_id: string}
   *   The active configuration.
   */
  public function getActiveConfig(): array {
    $config = $this->configFactory->get('nexi_xpay.settings');
    $env = $this->getEnvironment();

    /** @var array{base_url: string, api_key: string, merchant_id: string} $active */
    $active = $config->get($env);

    return [
      'base_url' => $active['base_url'],
      'api_key' => $active['api_key'],
      'merchant_id' => $active['merchant_id'],
    ];
  }

  /**
   * Returns the base URL for the Nexi API.
   */
  public function getBaseUrl(): string {
    $active = $this->getActiveConfig();
    return rtrim((string) $active['base_url'], '/');
  }

  /**
   * Returns the Nexi API key.
   *
   * @return string
   *   The API key.
   */
  public function getApiKey(): string {
    $active = $this->getActiveConfig();
    return (string) $active['api_key'];
  }

  /**
   * Returns TRUE if payloads should be logged.
   */
  public function shouldLogPayloads(): bool {
    return (bool) $this->configFactory->get('nexi_xpay.settings')->get('logging.log_payloads');
  }

  /**
   * {@inheritDoc}
   *
   * @throws \GuzzleHttp\Exception\GuzzleException
   */
  public function request(string $method, string $path, array $options = []): array {
    $this->loggerContext['method'] = strtoupper($method);
    $active = $this->getActiveConfig();
    $baseUrl = rtrim((string) $active['base_url'], '/');
    $url = $this->loggerContext['url'] = $baseUrl . '/' . ltrim($path, '/');
    $apiKey = $this->getApiKey();

    // IMPORTANT: don't throw on 4xx/5xx; we need body for diagnostics.
    $options += ['http_errors' => FALSE];

    // Normalize headers array for compliance to phpstan level 9.
    $extra_headers = $options['headers'] ?? [];
    $extra_headers = is_array($extra_headers) ? $extra_headers : [];

    $correlationId = $extra_headers['Correlation-Id'] ?? $this->uuid->generate();
    $correlationId = $this->loggerContext['correlationId'] =
      is_string($correlationId) && $correlationId !== '' ? $correlationId : $this->uuid->generate();

    $headers = $extra_headers + [
      'Accept' => 'application/json',
      'Correlation-Id' => $correlationId,
    ];

    // Defensive: if we're sending JSON, ensure content-type.
    if (isset($options['json']) && empty($headers['Content-Type'])) {
      $headers['Content-Type'] = 'application/json';
    }

    if ($apiKey !== '' && empty($headers['X-Api-Key'])) {
      $headers['X-Api-Key'] = $apiKey;
    }

    $options['headers'] = $headers;

    // Build context for logging.
    $this->logger->info('Nexi HTTP request', $this->loggerContext);
    $this->logger->debug(
      'Nexi HTTP request options',
      isset($options['json'])
        ? array_merge($this->loggerContext, ['payload' => $options['json']])
        : $this->loggerContext
    );

    // The request.
    $resp = $this->httpClient->request($method, $url, $options);

    $httpStatus = $this->loggerContext['httpStatus'] = $resp->getStatusCode();
    $headers = $resp->getHeaders();
    $body = trim((string) $resp->getBody());

    $this->logger->debug(
      'Nexi HTTP response',
      !empty($body)
        ? array_merge($this->loggerContext, ['body' => $body])
        : $this->loggerContext
    );

    return [
      'status' => $httpStatus,
      'headers' => $headers,
      'body' => $body,
    ];
  }

  /**
   * {@inheritDoc}
   *
   * @throws \GuzzleHttp\Exception\GuzzleException
   */
  public function requestJson(string $method, string $path, array $options = []): array {
    $resp = $this->request($method, $path, $options);
    $json = [];

    $body = trim($resp['body']);
    if ($body !== '') {
      try {
        $decoded = json_decode($body, TRUE, 512, JSON_THROW_ON_ERROR);
        $json = is_array($decoded) ? $decoded : [];
      }
      catch (\JsonException $e) {
        // Warning: recoverable; caller can still use raw body.
        $this->logger->warning(
          'Nexi JSON decode failed',
          array_merge(
            $this->loggerContext,
            [
              'method' => strtoupper($method),
              'endpoint' => $path,
              'error' => $e->getMessage(),
            ]
          )
        );
      }
    }

    $resp['json'] = $json;
    return $resp;
  }

  /**
   * {@inheritDoc}
   *
   * @throws \GuzzleHttp\Exception\GuzzleException
   */
  public function getOrder(string $orderId, array $options = []): array {
    $orderId = trim($orderId);

    $nexiOrder = $this->requestJson(
      'GET',
      '/orders/' . rawurlencode($orderId),
      array_merge($options, ['timeout' => 8, 'connect_timeout' => 5])
    );

    return $nexiOrder;
  }

}
