<?php

declare(strict_types=1);

namespace Drupal\nexi_xpay\Service;

/**
 * The interface defines the client for Nexi Xpay API.
 */
interface NexiXpayClientInterface {

  /**
   * Generic request helper. Mode plugins decide endpoints/payload.
   *
   * @param string $method
   *   HTTP method.
   * @param string $path
   *   Path relative to the API root.
   * @param array<string, mixed> $options
   *   Additional options to pass to curl_exec().
   *
   * @return array{status:int, headers:array<string, array<int, string>>, body:string}
   *   {status:int, headers:array, body:string}
   */
  public function request(string $method, string $path, array $options = []): array;

  /**
   * Searches for an order and returns its details.
   *
   * Nexi: GET /orders/{orderId}.
   * For more information about orders, please refer to the dedicated page
   * https://developer.nexi.it/en/api/get-orders-orderId .
   *
   * @param string $orderId
   *   Nexi Order ID.
   * @param array<string, mixed> $options
   *   Additional options to pass to curl_exec().
   *
   * @return array{status:int, headers:array<string, array<int, string>>, body:string, json:array<array-key, mixed>}
   *   {status:int, headers:array, body:string, json:array}
   */
  public function getOrder(string $orderId, array $options = []): array;

  /**
   * Returns the current environment (dev/prod).
   */
  public function getEnvironment(): string;

  /**
   * Like request(), but JSON-decodes the body.
   *
   * @param string $method
   *   HTTP method.
   * @param string $path
   *   Path relative to the API root.
   * @param array<string, mixed> $options
   *   Additional options to pass to curl_exec().
   *
   * @return array{status:int, headers:array<string, array<int, string>>, body:string, json:array<array-key, mixed>}
   *   {status:int, headers:array, body:string, json:array}
   */
  public function requestJson(string $method, string $path, array $options = []): array;

  /**
   * Do we want to log payloads?
   */
  public function shouldLogPayloads(): bool;

}
