<?php

declare(strict_types=1);

namespace Drupal\nexi_xpay\Service;

use Drupal\nexi_xpay\Entity\NexiXpayTransactionInterface;
use Drupal\nexi_xpay\Plugin\XpayMode\NexiXpayModeInterface;
use Drupal\nexi_xpay\Plugin\XpayMode\NexiXpayModeManager;
use Drupal\nexi_xpay\Value\ModeHandleResult;
use Drupal\nexi_xpay\Value\ModeStartResult;
use Psr\Log\LoggerInterface;
use Symfony\Component\HttpFoundation\Request;

/**
 * Nexi XPay payment flow manager.
 *
 * This service is the main entry point used by controllers and other
 * integration code to execute payment operations for a transaction.
 *
 * It delegates the actual implementation to a NexiXpay mode plugin selected
 * from the transaction configuration (e.g. Hosted Payment Page).
 *
 * The manager itself is intentionally thin:
 * - it resolves the correct mode plugin for the given transaction;
 * - it forwards the call to the plugin (start/return/notify/page builders);
 * - it provides a stable API for the rest of the module, keeping controllers
 *   agnostic of gateway-specific details.
 *
 * By centralizing mode resolution here, new payment modes can be added without
 * changing controllers or transaction handling logic.
 */
final readonly class NexiXpayManager implements NexiXpayManagerInterface {

  /**
   * The constructor.
   *
   * @param NexiXpayClientInterface $client
   *   The NexiXpay client.
   * @param \Drupal\nexi_xpay\Plugin\XpayMode\NexiXpayModeManager $modeManager
   *   The NexiXpay mode manager.
   * @param \Psr\Log\LoggerInterface $logger
   *   The logger interface.
   */
  public function __construct(
    private NexiXpayClientInterface $client,
    private NexiXpayModeManager $modeManager,
    private LoggerInterface $logger,
  ) {}

  /**
   * This method returns the mode plugin.
   *
   * @param \Drupal\nexi_xpay\Entity\NexiXpayTransactionInterface $transaction
   *   The transaction entity.
   *
   * @return \Drupal\nexi_xpay\Plugin\XpayMode\NexiXpayModeInterface
   *   The mode plugin.
   *
   * @throws \Drupal\Component\Plugin\Exception\PluginException
   */
  private function mode(NexiXpayTransactionInterface $transaction): NexiXpayModeInterface {
    $pluginId = $transaction->getMode();

    /** @var \Drupal\nexi_xpay\Plugin\XpayMode\NexiXpayModeInterface $plugin */
    $plugin = $this->modeManager->createInstance($pluginId, [
      'client' => $this->client,
      'logger' => $this->logger,
    ]);

    return $plugin;
  }

  /**
   * {@inheritDoc}
   *
   * @throws \Drupal\Component\Plugin\Exception\PluginException
   */
  public function startPayment(NexiXpayTransactionInterface $transaction, Request $request): ModeStartResult {
    return $this->mode($transaction)->startPayment($transaction, $request);
  }

  /**
   * {@inheritDoc}
   *
   * @throws \Drupal\Component\Plugin\Exception\PluginException
   */
  public function handleReturn(NexiXpayTransactionInterface $transaction, Request $request): ModeHandleResult {
    return $this->mode($transaction)->handleReturn($transaction, $request);
  }

  /**
   * {@inheritDoc}
   *
   * @throws \Drupal\Component\Plugin\Exception\PluginException
   */
  public function handleNotify(NexiXpayTransactionInterface $transaction, Request $request): ModeHandleResult {
    return $this->mode($transaction)->handleNotify($transaction, $request);
  }

  /**
   * {@inheritDoc}
   *
   * @throws \Drupal\Component\Plugin\Exception\PluginException
   */
  public function buildPayPage(NexiXpayTransactionInterface $transaction, Request $request): array {
    return $this->mode($transaction)->buildPayPage($transaction, $request);
  }

  /**
   * {@inheritDoc}
   *
   * @throws \Drupal\Component\Plugin\Exception\PluginException
   */
  public function buildReturnPage(NexiXpayTransactionInterface $transaction, Request $request, ModeHandleResult $handle): array {
    return $this->mode($transaction)->buildReturnPage($transaction, $request, $handle);
  }

}
