<?php

declare(strict_types=1);

namespace Drupal\nexi_xpay\Service;

use Drupal\nexi_xpay\Entity\NexiXpayTransactionInterface;
use Drupal\nexi_xpay\Event\NexiXpayEvents;
use Drupal\nexi_xpay\Event\TransactionStatusChangeEvent;
use Psr\Log\LoggerInterface;
use Symfony\Contracts\EventDispatcher\EventDispatcherInterface;

/**
 * Updates the transaction status.
 */
final class TransactionStatusUpdater {

  /**
   * The logger context.
   *
   * @var array{amount?: int, body?: string, correlationId?: string, currentStatus?: string, currency?: string, endpoint?: string, error?: string, eventId?: string, httpStatus?: int, method?: string, mode?: string, newStatus?: string|null, payload?: string, operationResult?: string, orderId?: string, outcome?: string, transaction?: int, url?: string}
   */
  protected array $loggerContext;

  /**
   * The constructor.
   *
   * @param \Symfony\Contracts\EventDispatcher\EventDispatcherInterface $dispatcher
   *   The event dispatcher.
   * @param \Drupal\nexi_xpay\Service\Logger $logger
   *   The logger.
   */
  public function __construct(
    private readonly EventDispatcherInterface $dispatcher,
    private readonly LoggerInterface $logger,
  ) {
    $this->loggerContext = $this->logger->setupContext();
  }

  /**
   * Sets the transaction status if it changed.
   *
   * @param \Drupal\nexi_xpay\Entity\NexiXpayTransactionInterface $transaction
   *   The transaction entity.
   * @param string $newStatus
   *   The new status.
   * @param array<string, mixed> $context
   *   Extra context information for callers (debug/logging).
   *
   * @return bool
   *   TRUE if the status changed.
   *
   * @throws \Drupal\Core\Entity\EntityStorageException
   */
  public function setStatusIfChanged(NexiXpayTransactionInterface $transaction, string $newStatus, array $context = []): bool {
    $currentStatus = $this->loggerContext['currentStatus'] = $transaction->getStatus();
    $this->loggerContext['newStatus'] = $newStatus;
    $this->loggerContext['transaction'] = (int) $transaction->id();

    if ($currentStatus === $newStatus) {
      return FALSE;
    }
    $transaction->setStatus($newStatus);
    $transaction->save();

    $this->dispatcher->dispatch(
      new TransactionStatusChangeEvent($transaction, $currentStatus, $newStatus, $context),
      NexiXpayEvents::TRANSACTION_STATUS_CHANGE
    );

    $this->loggerContext['mode'] =
      isset($context['mode']) && is_string($context['mode']) ? $context['mode'] : '';
    $this->loggerContext['orderId'] =
      isset($context['orderId']) && is_string($context['orderId']) ? $context['orderId'] : '';

    $this->logger->info('Transaction status changed.', $this->loggerContext);

    return TRUE;
  }

}
