<?php

declare(strict_types=1);

namespace Drupal\nexi_xpay\Util;

/**
 * Small utility to generate secrets/ids with consistent formats.
 *
 * - Hex secrets (bin2hex(random_bytes()))
 * - Base36 ids: lowercase [0-9a-z], fixed length via left padding + truncation.
 * - Optional SHA-256 hashing helper for stored hashes.
 */
final class SecretGenerator {

  /**
   * Generate a random hex string.
   *
   * @param int $bytes
   *   Number of bytes of entropy.
   *
   * @return string
   *   Hex string length = $bytes * 2.
   *
   * @throws \Random\RandomException
   */
  public static function generateHex(int $bytes): string {
    if ($bytes <= 0) {
      throw new \InvalidArgumentException('Bytes must be > 0.');
    }
    return bin2hex(random_bytes($bytes));
  }

  /**
   * Generate a random hex secret and its hash.
   *
   * Secret is hex (bin2hex), hash is SHA-256 hex by default.
   *
   * @param int $bytes
   *   Number of bytes of entropy for the secret.
   * @param string $algo
   *   Hash algorithm. Defaults to 'sha256'.
   *
   * @return array<string, string>
   *   {secret: string, hash: string}
   *
   * @throws \Random\RandomException
   */
  public static function generateHexSecretWithHash(int $bytes, string $algo = 'sha256'): array {
    $secret = self::generateHex($bytes);
    $hash = hash($algo, $secret);
    return ['secret' => $secret, 'hash' => $hash];
  }

  /**
   * Generate a random base36 id.
   *
   * Output is lowercase [0-9a-z] with a fixed length using left padding (0)
   * and truncation.
   *
   * @param int $length
   *   Desired length (e.g. 18 for Nexi order_id).
   * @param int $bytesEntropy
   *   Entropy bytes used to generate the id (default 12 ~= 96 bits).
   *
   * @return string
   *   Lowercase [0-9a-z] string of exactly $length chars.
   *
   * @throws \Random\RandomException
   */
  public static function generateBase36Id(int $length, int $bytesEntropy = 12): string {
    if ($length <= 0) {
      throw new \InvalidArgumentException('Length must be > 0.');
    }
    if ($bytesEntropy <= 0) {
      throw new \InvalidArgumentException('Entropy bytes must be > 0.');
    }

    $hex = bin2hex(random_bytes($bytesEntropy));
    $base36 = base_convert($hex, 16, 36);

    // Ensure fixed length.
    return substr(str_pad($base36, $length, '0', STR_PAD_LEFT), 0, $length);
  }

}
