<?php

declare(strict_types=1);

namespace Drupal\nexi_xpay\Value;

/**
 * Result object returned by XPay mode handlers (return/notify).
 *
 * It describes the outcome of processing a callback from Nexi:
 * - optionally a new transaction status to apply, and
 * - an arbitrary context array with details useful for logging, auditing,
 *   and for building the return page.
 *
 * The controller/service layer uses this object to update the transaction
 * in an idempotent way and to render user-facing messages.
 */
final readonly class ModeHandleResult {

  /**
   * The constructor.
   *
   * @param string|null $newStatus
   *   Optional new status computed by the mode (e.g. paid, failed).
   *   NULL means "no status change".
   * @param string $message
   *   Message to display.
   * @param array<string, mixed> $context
   *   Extra context information for callers (debug/logging/state updater).
   */
  private function __construct(
    public ?string $newStatus,
    public string $message,
    public array $context = [],
  ) {}

  /**
   * The result is a status change.
   *
   * @param string $newStatus
   *   The new status.
   * @param string $message
   *   Message to display.
   * @param array<string, mixed> $context
   *   Extra context information for callers (debug/logging/state updater).
   *
   * @return static
   *   The class instance.
   */
  public static function status(string $newStatus, string $message = '', array $context = []): self {
    return new self($newStatus, $message, $context);
  }

  /**
   * The result is a noop.
   *
   * @param string $message
   *   Message to display.
   * @param array<string, mixed> $context
   *   Extra context information for callers (debug/logging/state updater).
   *
   * @return static
   *   The class instance.
   */
  public static function noop(string $message = '', array $context = []): self {
    return new self(NULL, $message, $context);
  }

}
