<?php

namespace Drupal\next_redirects\Plugin\rest\resource;

use Drupal\rest\Plugin\ResourceBase;
use Symfony\Component\HttpKernel\Exception\NotFoundHttpException;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\rest\ResourceResponse;
use Drupal\Core\Url;
use Drupal\Core\Cache\CacheBackendInterface;
use Drupal\Component\Utility\UrlHelper;
use Drupal\Core\Database\Connection;

/**
 * Creates a resource for providing redirect data for Next.js.
 *
 * @RestResource(
 *   id = "next_redirects",
 *   label = @Translation("Next.js redirects"),
 *   uri_paths = {
 *     "canonical" = "/next_redirects"
 *   }
 * )
 */
class NextRedirectsResource extends ResourceBase implements ContainerFactoryPluginInterface {

  /**
   * Active database connection.
   *
   * @var \Drupal\Core\Database\Connection
   */
  protected $database;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    $instance = parent::create($container, $configuration, $plugin_id, $plugin_definition);
    $instance->database = $container->get('database');
    return $instance;
  }

  /**
   * Responds to GET requests, returns a list of redirects.
   *
   * @return \Drupal\rest\ResourceResponse
   *   REST response object with redirect data.
   *
   * @throws \Symfony\Component\HttpKernel\Exception\NotFoundHttpException
   *   Throws NotFoundHttpException in case of error or if no redirects exist.
   */
  public function get() {
    $data = [];

    $query = $this->database->select('redirect', 'r');
    $query->fields('r', ['rid', 'redirect_source__path', 'redirect_source__query', 'redirect_redirect__uri', 'status_code']);
    $results = $query->execute()->fetchAll();

    foreach ($results as $record) {
      // Let's get the redirect source path and query.
      $redirect_source_path = '/' . $record->redirect_source__path;
      $redirect_source_query = !empty($record->redirect_source__query) ? unserialize($record->redirect_source__query) : NULL;
      $redirect_source_query_array = is_array($redirect_source_query) ? $redirect_source_query : [];
      
      // Sort the query array by keys, to have a consistent order for
      // comparison later.
      self::ksortRecursive($redirect_source_query_array);

      // Now let's get the redirect_to (destination) path, including query
      // string if exists.
      $uri = $record->redirect_redirect__uri;
      $url = Url::fromUri($uri)->toString(TRUE);
      $redirect_to_path = $url->getGeneratedUrl();
      $redirect_to_parts = parse_url($redirect_to_path);
      $redirect_to_query_array = [];
      
      if (isset($redirect_to_parts['query'])) {
        // $redirect_to_parts['query'] could be an encoded string like:
        // "f%25255B0%25255D=bundle%3Aarticle&f%25255B1%25255D=bundle%3Apage&f%25255B2%25255D=im_field_department%3A1046"
        // or already decoded like "x=2&y=3".
        // We need to decode it properly before parsing.
        $decoded_query = self::decodeString($redirect_to_parts['query']);
        parse_str($decoded_query, $redirect_to_query_array);
        // Sort the query array by keys, to have a consistent order for
        // comparison later.
        self::ksortRecursive($redirect_to_query_array);
      }
      
      // If the redirect_from and redirect_to are identical, we check whether
      // it has query strings and if they are identical too, we skip this redirect.
      // This is to avoid infinite loops in Next.js redirects config.
      // Example: redirect from "/some-path?x=2&y=3" to "/some-path?y=3&x=2"
      // or from "/some-path" to "/some-path".
      if ($redirect_source_path == $redirect_to_path) {
        // Paths are identical and don't have query strings, skip this redirect.
        if (empty($redirect_source_query_array) && empty($redirect_to_query_array)) {
          continue;
        }
        // Both have query strings, compare them as arrays. If identical, skip this
        // redirect.
        elseif ($redirect_source_query_array == $redirect_to_query_array) {
          // Both paths and query strings are identical, skip this redirect.
          continue;
        }
      }

      // 'source' will only contain the path, query strings will be in 'has'
      // array.
      $redirect_item_data = [
        'source' => $redirect_source_path,
        'destination' => $redirect_to_path,
        'permanent' => $record->status_code == 301,
      ];

      // If there are query strings in the source, add them to 'has' array.
      // @see https://nextjs.org/docs/app/api-reference/config/next-config-js/redirects#header-cookie-and-query-matching
      if ($redirect_source_query_array) {
        foreach ($redirect_source_query_array as $key => $value) {
          $redirect_item_data['has'][] = [
            'type' => 'query',
            'key' => $key,
            'value' => $value,
          ];  
        }
      }

      $data[] = $redirect_item_data;
    }

    if ($data) {
      $response = new ResourceResponse($data, 200);
      $response->getCacheableMetadata()->setCacheMaxAge(CacheBackendInterface::CACHE_PERMANENT);
      $response->getCacheableMetadata()->addCacheTags(['redirect_list']);
      return $response;
    }
    throw new NotFoundHttpException(t("Can't find any redirects."));
  }

  /**
   * Decode a URL-encoded string up to 5 times.
   *
   * @param string $string
   *   The URL-encoded string to decode.
   *
   * @return string
   *   The fully decoded string.
   */
  protected static function decodeString(string $string) {
    // Decode up to 5 times, just in case.
    for ($i = 0; $i < 5; $i++) {
      $string = urldecode($string);
      if ($string === urldecode($string)) {
        break;
      }
    }
    return $string;
  }

  /**
   * Recursively sort an array by its keys.
   *
   * @param array $array
   *   The array to sort.
   */
  protected static function ksortRecursive(array &$array) {
    foreach ($array as &$value) {
      if (is_array($value)) {
        // Recursively call for nested arrays.
        self::ksortRecursive($value);
      }
    }
    // Sort the current array by key.
    ksort($array);
  }
}
