<?php

namespace Drupal\next_redirects\Plugin\rest\resource;

use Drupal\rest\Plugin\ResourceBase;
use Symfony\Component\HttpKernel\Exception\NotFoundHttpException;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\rest\ResourceResponse;
use Drupal\Core\Cache\CacheBackendInterface;

/**
 * Creates a resource for providing redirect data for Next.js.
 *
 * @RestResource(
 *   id = "next_redirects",
 *   label = @Translation("Next.js redirects"),
 *   uri_paths = {
 *     "canonical" = "/next_redirects"
 *   }
 * )
 */
class NextRedirectsResource extends ResourceBase implements ContainerFactoryPluginInterface {

  /**
   * Active database connection.
   *
   * @var \Drupal\Core\Database\Connection
   */
  protected $database;

  /**
   * The cache backend.
   *
   * @var \Drupal\Core\Cache\CacheBackendInterface
   */
  protected $cache;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    $instance = parent::create($container, $configuration, $plugin_id, $plugin_definition);
    $instance->database = $container->get('database');
    $instance->cache = $container->get('cache.default');
    return $instance;
  }

  /**
   * Responds to GET requests, returns a list of redirects.
   *
   * @return \Drupal\rest\ResourceResponse
   *   REST response object with redirect data.
   *
   * @throws \Symfony\Component\HttpKernel\Exception\NotFoundHttpException
   *   Throws NotFoundHttpException in case of error or if no redirects exist.
   */
  public function get() {
    $cid = 'next_redirects:processed_data';
    $cache_tags = ['redirect_list', 'path_alias_list'];

    // Try to get processed data from cache first.
    $cache = $this->cache->get($cid);
    if ($cache && !empty($cache->data)) {
      $data = $cache->data;
    }
    else {
      // Cache miss - process redirects.
      $data = $this->processRedirects();

      // Cache the processed data permanently, tagged for invalidation.
      $this->cache->set($cid, $data, CacheBackendInterface::CACHE_PERMANENT, $cache_tags);
    }

    if ($data) {
      $response = new ResourceResponse($data, 200);
      $cacheable_metadata = $response->getCacheableMetadata();
      $cacheable_metadata->setCacheMaxAge(CacheBackendInterface::CACHE_PERMANENT);
      $cacheable_metadata->addCacheTags($cache_tags);
      return $response;
    }
    throw new NotFoundHttpException($this->t("Can't find any redirects."));
  }

  /**
   * Process all redirects from the database.
   *
   * @return array
   *   Array of processed redirect data.
   */
  protected function processRedirects() {
    $data = [];

    $query = $this->database->select('redirect', 'r');
    $query->fields('r', ['rid', 'redirect_source__path', 'redirect_source__query', 'redirect_redirect__uri', 'status_code']);
    $results = $query->execute()->fetchAll();

    // Collect all internal destination paths that need alias lookup in a
    // single pass.
    $paths_to_lookup = [];

    foreach ($results as $record) {
      $uri = $record->redirect_redirect__uri;
      // Remove query string for subsequent alias lookup.
      $uri = strtok($uri, '?');
      $uri = trim($uri);
      // Check if the URI starts with a supported internal scheme.
      foreach (['internal:', 'entity:'] as $scheme) {
        if (strpos($uri, $scheme) === 0) {
          // Remove scheme prefix.
          $path = substr($uri, strlen($scheme));
          // If $path does not start with a slash, add it.
          if (strpos($path, '/') !== 0) {
            $path = '/' . $path;
          }

          // Add to paths to resolve.
          $paths_to_lookup[$record->rid] = $path;
          break;
        }
      }
    }

    // Bulk load all path aliases at once using chunked database queries.
    // We opted for getting aliases this way instead of using the Drupal's URL
    // system (e.g. `Url::fromUri()`) because that would involve bootstrapping
    // much more of Drupal, impacting performance significantly when processing
    // a large number of redirects.
    // We are also chunking the queries to avoid performance issues with large
    // `IN` clauses.
    $alias_results = [];
    if (!empty($paths_to_lookup)) {
      $unique_paths = array_unique($paths_to_lookup);
      // Chunk the paths to avoid IN clause performance issues with large arrays.
      // Using 500 as a conservative, safe limit.
      $chunk_size = 500;
      $chunks = array_chunk($unique_paths, $chunk_size);

      foreach ($chunks as $chunk) {
        // Query path_alias table in chunks for bulk lookup.
        $query = $this->database->select('path_alias', 'pa');
        $query->fields('pa', ['path', 'alias']);
        $query->condition('pa.path', $chunk, 'IN');
        // Only published aliases.
        $query->condition('pa.status', 1);
        $chunk_results = $query->execute()->fetchAllAssoc('path');
        $alias_results = array_merge($alias_results, $chunk_results);
      }
    }

    // Process all redirects using pre-loaded aliases.
    foreach ($results as $record) {
      // Let's get the redirect source path and query.
      $redirect_source_path = '/' . $record->redirect_source__path;
      $redirect_source_query = !empty($record->redirect_source__query) ? unserialize($record->redirect_source__query, ['allowed_classes' => FALSE]) : NULL;
      $redirect_source_query_array = is_array($redirect_source_query) ? $redirect_source_query : [];

      // Sort the query array by keys, to have a consistent order for
      // comparison later.
      $this->ksortRecursive($redirect_source_query_array);

      // Redirect URI from `redirect` table, with possible internal scheme and
      // query strings.
      $redirect_destination_uri = $record->redirect_redirect__uri;
      $redirect_destination_parts = parse_url($redirect_destination_uri);
      $redirect_destination_query_string = $redirect_destination_parts['query'] ?? [];

      // Use the pre-processed destination path (without scheme or alias) to
      // look up its alias in the pre-loaded results.
      $redirect_destination_path = $paths_to_lookup[$record->rid] ?? NULL;
      if ($redirect_destination_path !== NULL) {
        // Check for alias.
        if (isset($alias_results[$redirect_destination_path])) {
          $redirect_destination_path = $alias_results[$redirect_destination_path]->alias;
          if ($redirect_destination_query_string) {
            $redirect_destination_path .= '?' . $redirect_destination_query_string;
          }
        }
      }
      else {
        // External URL or unsupported scheme - use as is.
        $redirect_destination_path = $redirect_destination_uri;
      }

      $redirect_destination_query_array = [];

      if ($redirect_destination_query_string) {
        // $redirect_destination_parts['query'] could be an encoded string like:
        // "f%25255B0%25255D=bundle%3Aarticle&f%25255B1%25255D=bundle%3Apage&f%25255B2%25255D=im_field_department%3A1046"
        // or already decoded like "x=2&y=3".
        // We need to decode it properly before parsing.
        $decoded_query = $this->decodeString($redirect_destination_query_string);
        parse_str($decoded_query, $redirect_destination_query_array);
        // Sort the query array by keys, to have a consistent order for
        // comparison later.
        $this->ksortRecursive($redirect_destination_query_array);
      }
      // If the redirect_from and redirect_to are identical, we check whether
      // it has query strings and if they are identical too, we skip this redirect .
      // This is to avoid infinite loops in Next . js redirects config .
      // Example: redirect from "/some-path?x=2&y=3" to "/some-path?y=3&x=2"
      // or from "/some-path" to "/some-path" .
      if ($redirect_source_path == $redirect_destination_path) {
        // Paths are identical and don't have query strings, skip this redirect.
        if (empty($redirect_source_query_array) && empty($redirect_destination_query_array)) {
          continue;
        }
        // Both have query strings, compare them as arrays. If identical, skip this
        // redirect.
        elseif ($redirect_source_query_array == $redirect_destination_query_array) {
          // Both paths and query strings are identical, skip this redirect.
          continue;
        }
      }
      // 'source' will only contain the path, query strings will be in 'has'
      // array.
      $redirect_item_data = [
        'source' => $redirect_source_path,
        'destination' => $redirect_destination_path,
        'permanent' => $record->status_code == 301,
      ];

      // If there are query strings in the source, add them to 'has' array.
      // @see https://nextjs.org/docs/app/api-reference/config/next-config-js/redirects#header-cookie-and-query-matching
      if ($redirect_source_query_array) {
        foreach ($redirect_source_query_array as $key => $value) {
          $redirect_item_data['has'][] = [
            'type' => 'query',
            'key' => $key,
            'value' => $value,
          ];
        }
      }

      $data[] = $redirect_item_data;
    }

    return $data;
  }

  /**
   * Decode a URL-encoded string up to 5 times.
   *
   * @param string $string
   *   The URL-encoded string to decode.
   *
   * @return string
   *   The fully decoded string.
   */
  protected function decodeString(string $string) {
    // Decode up to 5 times, just in case.
    for ($i = 0; $i < 5; $i++) {
      $string = urldecode($string);
      if ($string === urldecode($string)) {
        break;
      }
    }
    return $string;
  }

  /**
   * Recursively sort an array by its keys.
   *
   * @param array $array
   *   The array to sort.
   */
  protected function ksortRecursive(array &$array) {
    foreach ($array as &$value) {
      if (is_array($value)) {
        // Recursively call for nested arrays.
        $this->ksortRecursive($value);
      }
    }
    // Sort the current array by key.
    ksort($array);
  }

}
